package cn.com.duiba.service.impl;

import cn.com.duiba.api.bo.credits.CreditsRequest;
import cn.com.duiba.api.bo.credits.CreditsResponse;
import cn.com.duiba.api.bo.subcredits.SubCreditsResultMsgDto;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.order.center.api.dto.CreditsCallbackMessage;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.ConsumerCreditsLogSaveDto;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.enums.CreditsLogChangeTypeEnum;
import cn.com.duiba.thirdparty.enums.CreditsLogStatusEnum;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created by fangdong on 2018/10/26
 */
@Component
public class CompositeCreditsLogExtractor implements CreditsLogExtractor {

    @Resource
    private List<CreditsLogExtractor> extractors;

    @Override
    public boolean supports(CreditsRequest request, CreditsResponse response) {
        return this.getExtractor(request, response) != null;
    }

    @Override
    public ConsumerCreditsLogSaveDto extract(CreditsRequest request, CreditsResponse response) {
        CreditsLogExtractor handler = this.getExtractor(request, response);
        if (handler != null) {
            return handler.extract(request, response);
        }

        return null;
    }

    private CreditsLogExtractor getExtractor(CreditsRequest request, CreditsResponse response) {
        for (CreditsLogExtractor extractor : extractors) {
            if (extractor.supports(request, response)) {
                return extractor;
            }
        }

        return null;
    }
}


interface CreditsLogExtractor {

    boolean supports(CreditsRequest request, CreditsResponse response);

    ConsumerCreditsLogSaveDto extract(CreditsRequest request, CreditsResponse response);

}

abstract class AbstractCreditsLogExtractor implements CreditsLogExtractor {
    private static final Logger log = LoggerFactory.getLogger(AbstractCreditsLogExtractor.class);

    @Override
    public boolean supports(CreditsRequest request, CreditsResponse response) {
        return this.supportsInternal(request, response);
    }

    @Override
    public ConsumerCreditsLogSaveDto extract(CreditsRequest request, CreditsResponse response) {
        ConsumerCreditsLogSaveDto creditsLog = this.extractInternal(request, response);
        if(creditsLog == null){
            return null;
        }

        Map<String, String> params = this.extractRequestParams(request);

        if (MapUtils.isEmpty(params)) {
            return null;
        }

        creditsLog.setOrderNum(params.get("orderNum"));
        creditsLog.setBizType(params.get("type"));
        if (params.get("credits") == null) {
            removeAppSecret(request);
//            log.info("积分记录获取credits为null, request={}", JSON.toJSONString(request));
            return null;
        }
        creditsLog.setCredits(Long.parseLong(params.get("credits")));
        creditsLog.setDescription(params.get("description"));

        return creditsLog;
    }

    private void removeAppSecret(CreditsRequest request) {
        if(request != null && request instanceof SubCreditsMsgWrapper){
            SubCreditsMsgWrapper subCreditsMsgWrapper = (SubCreditsMsgWrapper)request;
            if(subCreditsMsgWrapper.getSubCreditsMsg() != null){
                subCreditsMsgWrapper.getSubCreditsMsg().setAppSecret(null);
            }
        }
    }

    protected abstract boolean supportsInternal(CreditsRequest request, CreditsResponse response);

    protected abstract Map<String, String> extractRequestParams(CreditsRequest request);

    protected abstract ConsumerCreditsLogSaveDto extractInternal(CreditsRequest request, CreditsResponse response);
}

@Component
class MqSubCreditsLogExtractor extends AbstractCreditsLogExtractor {

    @Override
    public boolean supportsInternal(CreditsRequest request, CreditsResponse response) {
        return request instanceof SubCreditsMsgWrapper && response instanceof SubCreditsResultMsgDto;
    }

    @Override
    protected Map<String, String> extractRequestParams(CreditsRequest request) {
        SubCreditsMsgWrapper req = (SubCreditsMsgWrapper) request;
        return req.getSubCreditsMsg().getAuthParams();
    }

    @Override
    protected ConsumerCreditsLogSaveDto extractInternal(CreditsRequest request, CreditsResponse response) {
        SubCreditsMsgWrapper req = (SubCreditsMsgWrapper) request;
        SubCreditsResultMsgDto resp = (SubCreditsResultMsgDto) response;
        if(req.getSubCreditsMsg().getRelationType() == null){
            return null;
        }

        ConsumerCreditsLogSaveDto creditsLog = new ConsumerCreditsLogSaveDto();
        creditsLog.setConsumerId(req.getSubCreditsMsg().getConsumerId());
        creditsLog.setAppId(req.getSubCreditsMsg().getAppId());
        creditsLog.setChangeType(CreditsLogChangeTypeEnum.SUB);
        creditsLog.setLogStatus(CreditsLogStatusEnum.SUCCESS);
        creditsLog.setOrderTime(new Date(req.getSubCreditsMsg().getTimestamp()));
        creditsLog.setDeveloperBizId(resp.getBizId());
        creditsLog.setBizId(req.getSubCreditsMsg().getBizId());
        creditsLog.setRelationType(req.getSubCreditsMsg().getRelationType().getMsg());
        creditsLog.setRelationId(req.getSubCreditsMsg().getRelationId());

        return creditsLog;
    }

}

@Component
class RpcSubCreditsLogExtractor extends AbstractCreditsLogExtractor {

    @Override
    public boolean supportsInternal(CreditsRequest request, CreditsResponse response) {
        return request instanceof CreditsMessage && response instanceof CreditsCallbackMessage;
    }

    @Override
    protected Map<String, String> extractRequestParams(CreditsRequest request) {
        CreditsMessage req = (CreditsMessage) request;
        return req.getAuthParams();
    }

    @Override
    protected ConsumerCreditsLogSaveDto extractInternal(CreditsRequest request, CreditsResponse response) {
        CreditsMessage req = (CreditsMessage) request;
        CreditsCallbackMessage resp = (CreditsCallbackMessage) response;


        ConsumerCreditsLogSaveDto creditsLog = new ConsumerCreditsLogSaveDto();
        creditsLog.setConsumerId(Long.valueOf(req.getConsumerId()));
        creditsLog.setAppId(Long.valueOf(req.getAppId()));
        creditsLog.setChangeType(CreditsLogChangeTypeEnum.SUB);
        creditsLog.setLogStatus(CreditsLogStatusEnum.SUCCESS);
        creditsLog.setOrderTime(new Date(req.getTimestamp()));
        creditsLog.setDeveloperBizId(JSON.parseObject(resp.getMessage()).getString("bizId"));
        creditsLog.setBizId(req.getBizId());
        creditsLog.setRelationType(req.getRelationType());
        creditsLog.setRelationId(req.getRelationId());

        return creditsLog;
    }
}

@Component
class RpcAddCreditsLogExtractor extends AbstractCreditsLogExtractor {

    @Override
    public boolean supportsInternal(CreditsRequest request, CreditsResponse response) {
        return request instanceof CreditsMessageDto && response instanceof CreditsCallbackMessage;
    }

    @Override
    protected Map<String, String> extractRequestParams(CreditsRequest request) {
        CreditsMessageDto req = (CreditsMessageDto) request;
        return req.getAuthParams();
    }

    @Override
    protected ConsumerCreditsLogSaveDto extractInternal(CreditsRequest request, CreditsResponse response) {
        CreditsMessageDto req = (CreditsMessageDto) request;
        CreditsCallbackMessage resp = (CreditsCallbackMessage) response;

        ConsumerCreditsLogSaveDto creditsLog = new ConsumerCreditsLogSaveDto();
        creditsLog.setConsumerId(Long.valueOf(req.getConsumerId()));
        creditsLog.setAppId(Long.valueOf(req.getAppId()));
        creditsLog.setChangeType(CreditsLogChangeTypeEnum.ADD);
        creditsLog.setLogStatus(CreditsLogStatusEnum.SUCCESS);
        creditsLog.setOrderTime(new Date(req.getTimestamp()));
        String bizId = null;
        if (JSON.parseObject(resp.getMessage()).containsKey("bizId")) {
            bizId = JSON.parseObject(resp.getMessage()).getString("bizId");
        } else if (JSON.parseObject(resp.getMessage()).containsKey("supplierBizId")) {
            Map<String, String> params = this.extractRequestParams(request);
            if (MapUtils.isNotEmpty(params) && CreditsMessage.TYPE_EXCHANGE.equals(params.get("type"))) {
                bizId = JSON.parseObject(resp.getMessage()).getString("supplierBizId");
            }
        }
        creditsLog.setDeveloperBizId(bizId);
        creditsLog.setBizId(req.getBizId());
        creditsLog.setRelationType(req.getRelationType());
        creditsLog.setRelationId(req.getRelationId());

        return creditsLog;
    }

}
