package cn.com.duiba.tool.hsbc;

import cn.com.duiba.constant.hsbc.HsbcCertInitializer;
import cn.com.duiba.constant.hsbc.HsbcConfig;
import cn.com.duiba.consumer.center.api.dto.ConsumerExtraDto;
import cn.com.duiba.consumer.center.api.remoteservice.RemoteConsumerExtraService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.hutool.crypto.digest.DigestAlgorithm;
import cn.hutool.crypto.digest.Digester;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import io.jsonwebtoken.Jwts;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.Base64Utils;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author fja
 */
@Component
public class HsbcTool {

    private final static String SHA256_WITH_RSA = "SHA256withRSA";

    private final static String AES_PADDING_MODE = "AES/CBC/PKCS5Padding";

    private final static String RSA_PADDING_MODE = "RSA/ECB/PKCS1Padding";

    private final static String ALGORITHM_AES = "AES";

    private final static String JWT_TOKEN_PREFIX = "JWS ";

    /**
     * DUIBA rsa秘钥为2048bit
     * rsa 最大加密明文长度
     */
    private final static int DUIBA_RSA_MAX_ENCRYPT_LEN = 245;

    /**
     * 汇丰 rsa秘钥为2048bit
     * rsa 最大解密密文长度
     */
    private final static int HSBC_RSA_MAX_DECRYPT_LEN = 256;

    private final static int VECTOR_LENGTH = 16;

    private final static int KEY_LENGTH = 256;

    @Resource
    private HsbcConfig hsbcConfig;

//    @Resource
//    private HsbcCertInitializer hsbcCertInitializer;

    @Resource
    private RemoteConsumerExtraService remoteConsumerExtraService;


    private final Logger log = LoggerFactory.getLogger(this.getClass());


    /**
     * 签名, 返回base64编码的签名
     *
     * @param plainText          需要加密的明文
     * @param rsaDuibaPrivateKey 兑吧私钥
     * @return 签名
     * @throws Exception 异常
     */
    public String sign(String plainText, PrivateKey rsaDuibaPrivateKey) throws Exception {
        Signature sign = Signature.getInstance(SHA256_WITH_RSA);
        sign.initSign(rsaDuibaPrivateKey);
        sign.update(plainText.getBytes(StandardCharsets.UTF_8));
        return Base64.encodeBase64String(sign.sign());
    }


    /**
     * 校验签名
     *
     * @param origin        原签名
     * @param hsbcPublicKey 汇丰公钥
     * @param param         数据
     * @return true: 签名校验通过
     * @throws Exception -
     */
    public boolean verifySign(byte[] origin, PublicKey hsbcPublicKey, String param) throws Exception {
        Signature signature = Signature.getInstance(SHA256_WITH_RSA);
        signature.initVerify(hsbcPublicKey);
        signature.update(param.getBytes(StandardCharsets.UTF_8));
        return signature.verify(origin);
    }


    /**
     * AES加密
     *
     * @param plainText 明文
     * @param key       秘钥
     * @param vector    向量
     * @return base64编码后的加密字符串
     * @throws Exception -
     */
    public String aesEncrypt(String plainText, byte[] key, byte[] vector) throws Exception {
        if (vector.length % VECTOR_LENGTH != 0) {
            throw new IllegalArgumentException(String.format("向量长度异常 期望=%s 实际=%s", VECTOR_LENGTH, vector.length));
        }
        //每次都重新生成一个aesKey和aesIv
        IvParameterSpec ivParameterSpec = new IvParameterSpec(vector);
        SecretKey secretKeySpec = new SecretKeySpec(key, ALGORITHM_AES);
        Cipher cipher = Cipher.getInstance(AES_PADDING_MODE);
        cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, ivParameterSpec);
        byte[] bytes = cipher.doFinal(plainText.getBytes(StandardCharsets.UTF_8));
        return Base64Utils.encodeToString(bytes);
    }


    /**
     * AES 解密
     *
     * @param cipherText 加密字符串
     * @param key        秘钥
     * @param vector     向量
     * @return 明文
     * @throws Exception -
     */
    public String aesDecrypt(byte[] cipherText, byte[] key, byte[] vector) throws Exception {
        if (vector.length % VECTOR_LENGTH != 0) {
            throw new IllegalArgumentException(String.format("向量长度异常 期望=%s 实际=%s", VECTOR_LENGTH, vector.length));
        }
        IvParameterSpec ivParameterSpec = new IvParameterSpec(vector);
        SecretKey secretKeySpec = new SecretKeySpec(key, ALGORITHM_AES);
        Cipher cipher = Cipher.getInstance(AES_PADDING_MODE);
        cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, ivParameterSpec);
        byte[] bytes = cipher.doFinal(cipherText);
        return new String(bytes, StandardCharsets.UTF_8);
    }


    /**
     * 生成AES vector变量
     *
     * @return 变量
     * @throws NoSuchAlgorithmException -
     */
    public byte[] generateAesVector() throws NoSuchAlgorithmException {
        //生成16个字符的向量变量
//        SecureRandom sr = new SecureRandom();
        SecureRandom sr = SecureRandom.getInstance("NativePRNGNonBlocking");
        return sr.generateSeed(VECTOR_LENGTH);
    }


    /**
     * 生成AES key变量
     *
     * @return key
     * @throws NoSuchAlgorithmException -
     */
    public byte[] generateAesKey() throws NoSuchAlgorithmException {
        KeyGenerator kg = KeyGenerator.getInstance(ALGORITHM_AES);
        //AES 要求密钥长度为128位，192位，或者256位，默认128
        kg.init(KEY_LENGTH);
        //生成密钥
        SecretKey secretKey = kg.generateKey();
        //获得密钥的二进制编码形式
        return secretKey.getEncoded();
    }


    /**
     * RSA 加密
     *
     * @param plainText     明文
     * @param hsbcPublicKey 汇丰公钥
     * @return 加密字符串（aesKey aesIv）
     * @throws Exception -
     */
    public byte[] rsaEncrypt(byte[] plainText, PublicKey hsbcPublicKey) throws Exception {
        Cipher cipher = Cipher.getInstance(RSA_PADDING_MODE);
        cipher.init(Cipher.ENCRYPT_MODE, hsbcPublicKey);
        byte[] bytes = plainText.length > DUIBA_RSA_MAX_ENCRYPT_LEN ?
                doSplit(plainText, cipher, DUIBA_RSA_MAX_ENCRYPT_LEN) : cipher.doFinal(plainText);
        return bytes;
    }


    /**
     * RSA 解密
     *
     * @param cipherText         密文
     * @param rsaDuibaPrivateKey 兑吧rsa私钥
     * @return 明文
     * @throws Exception -
     */
    public byte[] rsaDecrypt(byte[] cipherText, PrivateKey rsaDuibaPrivateKey) throws Exception {
        Cipher cipher = Cipher.getInstance(RSA_PADDING_MODE);
        cipher.init(Cipher.DECRYPT_MODE, rsaDuibaPrivateKey);
        byte[] bytes = cipherText.length > HSBC_RSA_MAX_DECRYPT_LEN ?
                doSplit(cipherText, cipher, HSBC_RSA_MAX_DECRYPT_LEN) : cipher.doFinal(cipherText);
        return bytes;
    }


    /**
     * 明文超过了最大加密长度，分段加密
     * 密文超过了最大解密长度，分段解密
     *
     * @param cipherText 密文
     * @param cipher     -
     * @param threshold  长度阈值
     * @return -
     */
    private byte[] doSplit(byte[] cipherText, Cipher cipher, int threshold) throws Exception {
        int len = cipherText.length;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offSet = 0;
        byte[] cache;
        int i = 0;
        // 对数据分段解密
        while (len - offSet > 0) {
            if (len - offSet > threshold) {
                cache = cipher.doFinal(cipherText, offSet, threshold);
            } else {
                cache = cipher.doFinal(cipherText, offSet, len - offSet);
            }
            out.write(cache, 0, cache.length);
            i++;
            offSet = i * threshold;
        }
        out.close();
        return out.toByteArray();
    }


    /**
     * 截取域名
     *
     * @param url 请求链接
     * @return 域名
     */
    public String getHostName(String url) {
        return url.substring(0, url.indexOf('?'));
    }

    /**
     * 截取参数
     *
     * @param url 请求链接
     * @return 参数
     */
    public String getParamUrl(String url) {
        return url.substring(url.indexOf('?') + 1);
    }


    /**
     * 生成token
     *
     * @param claims jwt 声明
     * @param appId  应用id
     * @return token
     * @throws Exception 私钥读取异常
     */
    public String generateToken(Map<String, Object> claims, Long appId) throws Exception {
        Date expireDate = new Date(System.currentTimeMillis() + 12 * 60 * 60 * 1000);
        Map<String, Object> headers = Maps.newHashMap();
        headers.put("kid", "FB9B22FCAC8C839F");
        headers.put("typ", "JWT");
        headers.put("ver", "1.0");
        headers.put("alg", "PS256");
        String hsbcJwtKeyConf = findHsbcJwtKeyConfByAppId(appId);
        String token = JWT_TOKEN_PREFIX + Jwts.builder().setHeaderParams(headers).
                setClaims(claims).setExpiration(expireDate)
                .setId("11111111")
                .signWith(SignUtil.readPrivateFromPEMFile(hsbcJwtKeyConf))
                .compact();
        //log.info("[hsbc] jwt token =【{}】", token);
        return token;
    }


    /**
     * sha256
     *
     * @param plainText 明文
     * @return -
     */
    public String sha256(String plainText) {
        Digester sha256 = new Digester(DigestAlgorithm.SHA256);
        return sha256.digestHex(plainText);
    }


    /**
     * 根据appId查找对应的汇丰公钥
     *
     * @param appId 应用id
     * @return 公钥对象
     */
    public PublicKey findHsbcPublicKeyByAppId(Long appId) {
        PublicKey publicKey = hsbcConfig.getAppId2HsbcPublicKeyMap().get(appId);
        if (Objects.isNull(publicKey)) {
            throw new IllegalArgumentException(String.format("汇丰公钥查询失败，appId = %s", appId));
        }
        return publicKey;
    }


    /**
     * 根据appId查找对应的兑吧私钥
     *
     * @param appId 应用id
     * @return 私钥对象
     */
    public PrivateKey findDuibaPrivateKeyByAppId(Long appId) {
        PrivateKey privateKey = hsbcConfig.getAppId2RsaDuibaPrivateKeyMap().get(appId);
        if (Objects.isNull(privateKey)) {
            throw new IllegalArgumentException(String.format("兑吧查询失败，appId = %s", appId));
        }
        return privateKey;
    }


    /**
     * 根据appId查找对应的jwt秘钥配置
     *
     * @param appId 应用id
     * @return jwt秘钥配置
     */
    private String findHsbcJwtKeyConfByAppId(Long appId) {
        String jwtKeyConf = hsbcConfig.getAppId2JwtKeyConfMap().get(String.valueOf(appId));
        if (StringUtils.isBlank(jwtKeyConf)) {
            throw new IllegalArgumentException(String.format("汇丰jwt key配置查询失败，appId = %s", appId));
        }
        return jwtKeyConf;
    }


    /**
     * 根据appId查找对应环境的userInfoUrl
     *
     * @param appId 应用id
     * @return url
     */
    public String findUserInfoUrlByAppId(Long appId) {
        String userInfoUrl = hsbcConfig.getAppId2UserInfoUrlMap().get(String.valueOf(appId));
        if (Objects.isNull(userInfoUrl)) {
            throw new IllegalArgumentException(String.format("汇丰userInfoUrl查询失败，appId = %s", appId));
        }
        return userInfoUrl;
    }


    /**
     * 根据appId查找对应环境的pointTaskListUrl
     *
     * @param duibaAppId duiba应用id
     * @return url
     */
    public String findPointTaskListUrlByAppId(Long duibaAppId) {
        String pointTaskListUrl = hsbcConfig.getAppId2PointTaskListUrlMap().get(String.valueOf(duibaAppId));
        if (StringUtils.isBlank(pointTaskListUrl)) {
            throw new IllegalArgumentException(String.format("汇丰PointTaskListUrl查询失败，duibaAppId = %s", duibaAppId));
        }
        return pointTaskListUrl;
    }

    /**
     * 查询任务状态url
     *
     * @param duibaAppId 兑吧应用id
     * @return url
     */
    public String findPointTaskStatusUrlByAppId(Long duibaAppId) {
        String pointTaskStatusUrl = hsbcConfig.getAppId2PointTaskStatusUrlMap().get(String.valueOf(duibaAppId));
        if (StringUtils.isBlank(pointTaskStatusUrl)) {
            throw new IllegalArgumentException(String.format("汇丰pointTaskStatusUrl查询失败，duibaAppId = %s", duibaAppId));
        }
        return pointTaskStatusUrl;
    }


    /**
     * 根据consumerId去consumerExtra中获取partner_user_id
     *
     * @param consumerId 兑吧用户id
     * @return partner_user_id
     */
    @Deprecated
    public String getUidByConsumerId(Long consumerId) {
        if (Objects.isNull(consumerId)) {
            throw new IllegalArgumentException("[hsbc]获取uid失败 consumerId为空");
        }


        DubboResult<ConsumerExtraDto> dubboResult = remoteConsumerExtraService.findByConsumerId(consumerId);
        if (!dubboResult.isSuccess()) {
            throw new IllegalArgumentException(String.format("[hsbc]cid = %s 获取uid失败, remoteConsumerExtraService#findByConsumerId异常", consumerId));
        }


        ConsumerExtraDto consumerExtraDto = dubboResult.getResult();
        if (StringUtils.isBlank(consumerExtraDto.getJson())) {
            throw new IllegalArgumentException(String.format("[hsbc]cid = %s 获取uid失败， json字段为空", consumerId));
        }


        JSONObject extJson = JSON.parseObject(consumerExtraDto.getJson());
        String uid = extJson.getString("uid");
        if (StringUtils.isBlank(uid)) {
            throw new IllegalArgumentException(String.format("[hsbc]cid = %s 获取uid失败，json = %s", consumerId, consumerExtraDto.getJson()));
        }


        return uid;
    }

    /**
     * 汇丰资讯，签名生产
     */
    public void generateNewsSign(HttpRequest request) {
        String secretKey = hsbcConfig.getNewsSecretKey();
        String secretValue = hsbcConfig.getNewsSecretValue();
        String timestamp = Long.toString(System.currentTimeMillis());
        String signRaw = String.format("%s_%s", secretValue, timestamp);
        /**
         * ⽣成签名
         * package org.apache.commons.codec.digest;
         */
        String sign = DigestUtils.md5Hex(signRaw);
        // 请求头
        request.setHeader("timestamp", timestamp);
        request.setHeader("secretKey", secretKey);
        request.setHeader("sign", sign);
        log.info("汇丰-资讯签名参数，params={}", JSON.toJSONString(request.getAllHeaders()));
    }
}
