/**
 * Project Name:tuia-core-api<br>
 * File Name:RemoteAccountBackendService.java<br>
 * Package Name:cn.com.duiba.tuia.core.api.remoteservice<br>
 * Date:2016年8月23日上午11:11:53<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.core.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.tuia.core.api.dto.*;
import cn.com.duiba.tuia.core.api.dto.advertiser.req.ReqPageQueryAdvertiserDto;
import cn.com.duiba.tuia.core.api.dto.req.*;
import cn.com.duiba.tuia.core.api.dto.req.account.AccountDetailQueryDto;
import cn.com.duiba.tuia.core.api.dto.req.account.AccountQueryDto;
import cn.com.duiba.tuia.core.api.dto.req.account.ReqAccountEditDto;
import cn.com.duiba.tuia.core.api.dto.rsp.RspAderStatInfoDto;
import cn.com.duiba.tuia.core.api.dto.rsp.RspAppointmentInfo;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.web.bind.annotation.RequestMapping;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * ClassName: RemoteAccountBackendService <br/>
 * Function: 用户. <br/>
 * date: 2016年8月23日 上午11:11:53 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@AdvancedFeignClient
public interface RemoteAccountBackendService {

    /**
     * 注册.
     *
     * @param req the req
     * @param userType the user type
     * @return the dubbo result< long>
     */
    DubboResult<AccountDto> register(ReqRegisterDto req, int userType);

    /**
     * 创建账号.
     *
     * @param accountDto 账号信息
     * @param bindingOperator 绑定操作员
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> createAdvertiser(AccountDto accountDto, Long bindingOperator) throws BizException;

    /**
     * 重置密码.
     *
     * @param email 邮箱
     * @param newPasswd 新密码
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> resetPassword(String email, String newPasswd);

    /**
     * 更新邮箱审核状态.
     *
     * @param email the email
     * @return the dubbo result< boolean>
     */
    DubboResult<BaseAccountDto> updateEmailStatus(String email);

    /**
     * 账号审核.
     *
     * @param id 账号ID
     * @param checkType 审核类型
     * @param checker 审核人
     * @param checkerContact the checker contact
     * @param reason 原因
     * @return the int
     */
    DubboResult<Boolean> updateCheckStatus(long id, int checkType, String checker, String checkerContact, String reason);

    /**
     * 账号冻结与解冻.
     *
     * @param accoutId 广告主账户ID
     * @param status 冻结或解冻状态值
     * @param reason 冻结原因
     * @return the int
     */
    DubboResult<Boolean> updateFreezeStatus(long accoutId, int status, String reason);

    /**
     * 更新账号信息.
     *
     * @param updateAccountDto the update account dto
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> updateAccount(ReqUpdateAccountDto updateAccountDto);

    /**
     * 修改广告主信息.
     *
     * @param reqUpdateAdvertiserDto the req update advertiser dto
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> updateAdvertiser(ReqUpdateAdvertiserDto reqUpdateAdvertiserDto);

    /**
     * 更新审核材料.
     *
     * @param req the req
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> updateAuditData(ReqUpdateAuditData req);

    /**
     * 更新审核材料.
     *
     * @param reqUpdateAdvertiserAuditData the req update advertiser audit data
     * @return the dubbo result< boolean>
     */
    DubboResult<Boolean> updateAdvertiserAuditData(ReqUpdateAdvertiserAuditData reqUpdateAdvertiserAuditData);

    /**
     * 通过邮箱查询账号信息.
     *
     * @param email 邮箱
     * @return 账号信息
     */
    DubboResult<AccountDto> getAccountByEmail(String email);

    /**
     * 通过账号ID查询账号信息.
     *
     * @param accountId 账号ID
     * @return 账号信息
     */
    DubboResult<AccountDto> getAccountById(long accountId);

    /**
     * 判断账号是否是特殊代理商(直客)
     *
     * @param accountId 账号id
     * @return true 是，false 否
     */
    Boolean isSpecialAgent(Long accountId);

    /**
     * 分页查询广告主.
     *
     * @param req 分页查询请求参数
     * @return 广告主查询结果
     */
    @RequestMapping("/pageQuery1")
    DubboResult<PageDto<AdvertiserSumInfoDto>> pageQuery(ReqPageQueryAccount req);

    /**
     * 得到最新的审核记录. <br/>
     *
     * @param accountId the account id
     * @param checkType the check type
     * @return the account check record
     * @author ZFZ
     * @since JDK 1.6
     */
    DubboResult<AccountCheckRecordDto> getAccountCheckRecord(long accountId, int checkType);

    /**
     * 查询代理商帐号id旗下广告主帐号数量. <br/>
     *
     * @param agentAccountId the agent account id
     * @return the advertiser count
     * @author sunjiangrong
     * @since JDK 1.6
     */
    public DubboResult<Integer> getAdvertiserCount(long agentAccountId);

    /**
     * 根据ID列表获取账户基本信息. <br/>
     *
     * @param ids the ids
     * @return the dubbo result< list< base account dto>>
     * @author sunjiangrong
     * @since JDK 1.6
     */
    DubboResult<List<BaseAccountDto>> selectAccountBaseInfoByIds(List<Long> ids);

    /**
     * 查询代理商下(邮箱已验证、审核通过、未冻结的)广告主.
     * 该方法返回的只有基本的id、agentId、email和name，若需要全量的信息请使用selectAdvertisersByAgentId方法
     *
     * @param agentId 代理商ID
     * @return 广告主基本信息列表
     * @see RemoteAccountBackendService#selectAdvertisersByAgentId(java.lang.Long)
     */
    DubboResult<List<BaseAccountDto>> getAdvertisersByAgentId(Long agentId);

    /**
     * 查询广告主全量信息.
     *
     * @param agentId 代理商ID
     * @return 广告主信息列表
     */
    DubboResult<List<AccountDto>> selectAdvertisersByAgentId(Long agentId);

    /**
     * 通过eamils查询代理商ids <br/>.
     *
     * @param emails the emails
     * @return the agent ids
     * @since JDK 1.6
     */
    DubboResult<List<Long>> getAgentIds(List<String> emails);

    /**
     * 代理商首页查询广告主信息.
     *
     * @param agentId the agent id
     * @return the ader stat info
     */
    DubboResult<RspAderStatInfoDto> getAderStatInfo(Long agentId);

    /**
     * 申请预约信息保存
     *
     * @author chencheng
     * @param reqAppointmentInfo
     * @return
     */
    DubboResult<Boolean> appointmentApply(ReqAppointmentInfo reqAppointmentInfo);

    /**
     * 根据代理商名称模糊id列表
     * @param agentName  代理商名称
     * @return
     */
    DubboResult<List<Long>> getAgentIdsByName(String agentName);


    DubboResult<List<Long>> getAccountIdsByName(String advertiserName);

    /**
     * 通过客户名称模糊查询所有客户id
     *
     * @param accountName 需要搜索的客户名称
     * @return 搜索到的accountIds
     */
    List<Long> queryAllAccountIdsByName(String accountName);

    /**
     * 批量更新广告主(代理商)账号状态
     *
     * @param accountIds 代理商id
     * @param updateType 0:审核状态，1:冻结状态，3:邮箱状态
     * @param status     对应状态的值
     * @return
     */
    DubboResult<Boolean> updateAccountStatus(List<Long> accountIds, int updateType, int status);

    /**
     * 分页查询
     *
     * @param req 查询参数
     * @return
     */
    @RequestMapping("/pageQuery2")
    DubboResult<List<AdvertiserSumInfoDto>> pageQuery(ReqPageQueryAdvertiserDto req);

    List<AdvertiserSumInfoDto> pageQueryAccountList(ReqPageQueryAdvertiserDto req);

    Integer countPageQueryAccountList(ReqPageQueryAdvertiserDto req);

    /**
     * 分页查询计数
     *
     * @param req 查询参数
     * @return
     */
    DubboResult<Integer> getPageQueryAmount(ReqPageQueryAdvertiserDto req);

    /**
     * 批量查询广告主
     *
     * @param advertiserIds 广告主id
     * @return 广告主
     */
    DubboResult<List<AccountDto>> selectAdvertiserListByIds(List<Long> advertiserIds);

    /**
     * 根据账户ids和用户类型查询账户信息
     *
     * @param ids      查询的账号ids
     * @param userType 0-广告主，1协作员，2-代理商，详见cn.com.duiba.tuia.core.api.constant.AccountType，非必传
     * @return 查询的账号列表
     */
    List<AccountDto> selectByUserTypeAndIds(List<Long> ids, Integer userType);

    /**
     * 通过companyName和agentName查询广告主id
     *
     * @param companyName
     * @param agentCompanyName
     * @return
     */
    DubboResult<List<Long>> selectAdvertiserIdsByName(String companyName, String agentCompanyName);

    List<Long> selectIdsByCompanyAndAgentName(String companyName, String agentCompanyName);

    /**
     * 修改帐号信息
     * 
     * @param changeDto 修改的信息
     * @return true 修改成功 false 修改失败
     */
    DubboResult<Boolean> changeAccountInfo(ReqAccountEditDto changeDto);


    /**
     * 根据条件查询帐号id
     * @param queryDto 查询条件
     * @return 符合条件的帐号id
     */
    List<AccountDto> selectListByParam(AccountDetailQueryDto queryDto);
    
    /**
     * 根据条件查询帐号id
     * @param queryDto 查询条件
     * @return 符合条件的帐号id
     */
    List<Long> getAccountIds(AccountQueryDto queryDto);
    

    /**
     * 根据account id 修改account信息
     * 
     * @param accountDto 修改的account信息
     * @return 修改的记录数
     */
    int updateById(AccountDto accountDto) ;
    
    /**
     * 
     * selectAppointmentInfo:(查询客户预约信息). <br/>
     *
     * @author chencheng
     * @param dto
     * @return
     * @since JDK 1.8
     */
    List<RspAppointmentInfo> selectAppointmentInfo(QueryDto dto);
    
    /**
     * 
     * selectAppointmentInfoCount:(查询客户预约信息总数). <br/>
     *
     * @author chencheng
     * @param dto
     * @return
     * @since JDK 1.8
     */
    Integer selectAppointmentInfoCount(QueryDto dto);

    List<Long> getSpecialAgentIds();

    /**
     * 
     * createAgent:(创建下级代理商). <br/>
     *
     * @author chencheng
     * @param accountDto
     * @return
     * @since JDK 1.8
     */
    Boolean createAgent(AccountDto accountDto);

    /**
     * 
     * pageQueryAgent:(多级代理商分页查询). <br/>
     *
     * @author chencheng
     * @param req
     * @return
     * @since JDK 1.8
     */
    Integer pageQueryAgentCount(ReqPageQueryAccount req);
    
    /**
     * 
     * pageQueryAgent:(多级代理商分页查询). <br/>
     *
     * @author chencheng
     * @param req
     * @return
     * @since JDK 1.8
     */
    List<AdvertiserSumInfoDto> pageQueryAgent(ReqPageQueryAccount req);
    
    /**
     * 
     * getAgentNextAgentds:(获取代理商下所有的下级广告主(或代理商)id). <br/>
     *
     * @author chencheng
     * @param agentId
     * @return
     * @since JDK 1.8
     */
    List<Long> getAgentNextAdvertiserIds(Long agentId, AccountDetailQueryDto queryDto);
    
    /**
     * 
     * getAgentNextAdvertiser:(获取代理商下所有的下级广告主(或代理商)). <br/>
     *
     * @author chencheng
     * @param agentId
     * @return
     * @since JDK 1.8
     */
    List<AccountDto> getAgentNextAdvertiser(Long agentId, AccountDetailQueryDto queryDto);
    
    /**
     * 
     * selectOperatorMap:(查询广告主对应的操作员). <br/>
     *
     * @author chencheng
     * @param advertiserIdList
     * @return
     * @since JDK 1.8
     */
    Map<Long, AccountDto> selectOperatorMap(List<Long> advertiserIdList);

    Map<Long, Long> getAgentIdAndSpecialAdvertiserId(List<Long> accountId);

    List<AccountDto> selectNameByAdvertiser(List<Long> advertiserIds);

    Date firstLaunchTime(Long accountId);

    /**
     * 根据代理商等级和代理商公司名称查询代理商及其下级代理商的所有的广告主id
     *
     * @param accountLevel 代理商等级  0，1，2，3
     * @param companyName  代理商名称
     * @param email        代理商邮箱
     * @return 所查找的代理商下所有的广告主id
     */
    List<Long> selectAdvertiserIdUnderAgentLevelAndName(Integer accountLevel, String companyName, String email);

    /**
     * 根据代理商等级和代理商公司名称查询代理商及其下级代理商和所有的广告主id
     * 对比{@link RemoteAccountBackendService#selectAdvertiserIdUnderAgentLevelAndName(java.lang.Integer, java.lang.String, java.lang.String)} 的返回结果增加了代理商的id
     *
     * @param accountLevel 代理商等级  0，1，2，3
     * @param companyName  代理商名称
     * @param email        代理商邮箱
     * @return 所查找的代理商及其下级代理商的id和所有的广告主id
     */
    List<Long> selectAccountIdIdUnderAgentLevelAndName(Integer accountLevel, String companyName, String email);

    /**
     * 根据代理商/广告主id,类型，公司名称，邮箱等获取广告主或代理商信息
     *
     * @author chencheng
     * @param agentIds
     * @param agentIds
     * @param userType：0:广告主，2：代理商
     * @param name
     * @param email
     * @return
     */
    List<AccountDto> selectIdsByAccountIdNameEmail(List<Long> ids, List<Long> agentIds, int userType, String name, String email) ;


    /**
     * 根据账户id获取一级代理商和直客的帐号id
     *
     * @author lijian
     */
    Map<Long, Long> getOneAgentIdAndSpecialAdvertiserId(List<Long> accountIds);

    /**
     * 根据公司名称模糊查询广告主公司名称.<br>
     * @param companyName  公司名称
     */
    List<AccountDto> getNamesByCompanyName(String companyName);

    /**
     * 根据在传入的accountIds列表中，根据 companyName 模糊查询
     * @param accountIds
     * @param companyName
     * @return
     */
    List<AccountDto> selectByAccountIdsAndCompanyNameLike(List<Long> accountIds,String companyName);

    /**
     * 根据传入的accountIds列表，查询对应的经营范围信息
     * @param accountIds
     * @return
     */
    List<CompanyInfoDto> getListByAccountIds(List<Long> accountIds);


    /**
     * 广告主余额不足设置不再提醒
     * @param aeId
     */
    void advertiserBalanceNotTip(Long aeId);

    /**
     * 根据绑定手机号查询 广告主信息
     * @param bindphone
     * @return
     */
    List<AccountDto> selectByBindphone(String bindphone);

}

