/**
 * Project Name:media-biz<br>
 * File Name:SlotCacheServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月30日下午4:48:25<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.service.impl;

import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.SlotCacheDto;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.StrategyCacheDto;
import cn.com.duiba.tuia.media.api.utils.CacheKeyUtils;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.constants.SlotPictureSizeEnmu;
import cn.com.duiba.tuia.media.common.constants.SlotPictureSizeEnmu.PictureSize;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ShieldStrategyDAO;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.domain.ShieldStrategyDto;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.utils.StringTool;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: SlotCacheServiceImpl <br/>
 * Function: 广告位缓存层. <br/>
 * date: 2016年9月30日 下午4:48:25 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class SlotCacheService extends BaseCacheService {

    @Autowired
    private SlotDAO           slotDAO;

    @Autowired
    private ShieldStrategyDAO shieldStrategyDAO;

    /**
     * 获取广告位信息.
     *
     * @param slotId 广告位ID
     * @return 广告位信息
     * @throws TuiaMediaException the tuia media exception
     */
    public SlotCacheDto getSlot(long slotId) throws TuiaMediaException {
        String strSlot = redisClient.get(CacheKeyUtils.getSlotKey(slotId));
        SlotCacheDto slotCacheDto;
        if (StringUtils.isEmpty(strSlot)) {
            slotCacheDto = updateSlotById(slotId);
        } else {
            slotCacheDto = JSONObject.parseObject(strSlot, SlotCacheDto.class);
        }
        if (slotCacheDto == null) {
            // 广告位不存在
            throw new TuiaMediaException(ErrorCode.E0304008);
        }
        return slotCacheDto;
    }

    /**
     * 更新广告位信息.
     *
     * @param slotId 广告位ID
     */
    public void updateSlotCache(final long slotId) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                try {
                    updateSlotById(slotId);
                } catch (TuiaMediaException e) {
                    logger.error("updateSlotCache error", e);
                }

            }
        });
    }

    /**
     * 批量更新广告位信息.
     *
     * @param slotIds 广告位ID列表
     */
    public void updateBatchSlotCache(final List<Long> slotIds) {
        executorService.submit(new Runnable() {

            @Override
            public void run() {
                try {
                    if (CollectionUtils.isNotEmpty(slotIds)) {
                        for (Long slotId : slotIds) {
                            updateSlotById(slotId);
                        }
                    }

                } catch (TuiaMediaException e) {
                    logger.error("updateSlotCache error", e);
                }

            }
        });
    }

    /**
     * 执行查询广告位信息.
     *
     * @param slotId 广告位id
     * @return 广告位信息
     * @throws TuiaMediaException
     */
    public SlotCacheDto updateSlotById(long slotId) throws TuiaMediaException {
        SlotDto slotDto = slotDAO.selectById(slotId);
        SlotCacheDto slotCacheDto = null;
        if (slotDto != null) {
            if (slotDto.getPictureSize() != null) {
                PictureSize pictureSize = SlotPictureSizeEnmu.getPictureSizeById(slotDto.getPictureSize());
                if (pictureSize == null) {
                    // 广告位尺寸异常
                    logger.error(" the slot=[{}] pictureSizeId=[{}] non-existent", slotId, slotDto.getPictureSize());
                    return null;
                }
                String pictureType = null;
                if (SlotPictureSizeEnmu.SIZE_42.getSizeId() == slotDto.getPictureSize()) {
                    pictureType = SlotCacheDto.PICTURE_TYPE_ALPHA0;
                }
                slotCacheDto = new SlotCacheDto(slotId, slotDto.getSlotType(), pictureSize.getWidth(),
                                                pictureSize.getHeight(), slotDto.getStrategyId(), pictureType);
            } else {
                slotCacheDto = new SlotCacheDto(slotId, slotDto.getSlotType(), 0, 0, slotDto.getStrategyId(), null);
            }

            slotCacheDto.setValid(SlotDto.CHECK_STATUS_PASS == slotDto.getCheckStatus() && SlotDto.ENABLE_STATUS_OPEN == slotDto.getEnableStatus());
            slotCacheDto.setVisibleOfCloseButton(slotDto.isVisibleOfCloseButton());
            slotCacheDto.setVisibleOfIco(slotDto.isVisibleOfIco());
            slotCacheDto.setActivityTargetUrl(slotDto.getActivityTargetUrl());
            slotCacheDto.setSlotExposeLimit(slotDto.getSlotExposeLimit());
            slotCacheDto.setActivityExposeLimit(slotDto.getActivityExposeLimit());
            slotCacheDto.setSlotMsId(slotDto.getSlotMsId());
            redisClient.setex(CacheKeyUtils.getSlotKey(slotId), ONE_WEEK, JSONObject.toJSONString(slotCacheDto));
        }
        return slotCacheDto;
    }

    /**
     * 获取广告位的屏蔽策略信息.
     *
     * @param slotId 广告位ID
     * @return 广告位的屏蔽策略信息
     * @throws TuiaMediaException the tuia media exception
     */
    public StrategyCacheDto getStrategy(long slotId) throws TuiaMediaException {
        Long strategyId = getSlot(slotId).getStrategyId();
        if (strategyId == null) {
            // 该广告位没有屏蔽策略
            return null;
        }
        StrategyCacheDto strategyDto;

        String strategy = redisClient.get(CacheKeyUtils.getStrategyKey(strategyId));
        if (StringUtils.isEmpty(strategy)) {
            // 查询屏蔽策略信息
            strategyDto = updateStrategyById(strategyId);
        } else {
            strategyDto = JSONObject.parseObject(strategy, StrategyCacheDto.class);
        }
        return strategyDto;
    }

    /**
     * 更新屏蔽策略信息.
     *
     * @param slotId 广告位id
     * @return 屏蔽策略信息
     * @throws TuiaMediaException the tuia media exception
     */
    public StrategyCacheDto updateStrategy(long slotId) throws TuiaMediaException {
        Long strategyId = getSlot(slotId).getStrategyId();
        if (strategyId == null) {
            // 该广告位没有屏蔽策略, 返回null
            return null;
        }

        return updateStrategyById(strategyId);
    }

    /**
     * 执行查询屏蔽策略操作.
     *
     * @param strategyId 屏蔽策略ID
     * @return the strategy dto
     * @throws TuiaMediaException the tuia media exception
     */
    public StrategyCacheDto updateStrategyById(long strategyId) throws TuiaMediaException {
        StrategyCacheDto strategyDto = null;

        ShieldStrategyDto shieldStrategyDto = shieldStrategyDAO.selectById(strategyId);
        if (shieldStrategyDto != null) {
            strategyDto = new StrategyCacheDto(shieldStrategyDto.getId(), StringTool.getStringListByStr(shieldStrategyDto.getShieldIndustries()),
                                               StringTool.getStringListByStr(shieldStrategyDto.getShieldUrls()), StringTool.getStringListByStr(shieldStrategyDto.getAdvertTagNums()));
            // 塞入缓存
            redisClient.setex(CacheKeyUtils.getStrategyKey(strategyId), ONE_WEEK, JSONObject.toJSONString(strategyDto));
        } else {
            logger.error("the strategyId=[{}] is non-existent!", strategyId);
        }

        return strategyDto;
    }

    /**
     * 删除缓存.
     *
     * @param strategyId the strategy id
     */
    public void deleteStrategyCache(long strategyId) {
        // 删除缓存
        redisClient.del(CacheKeyUtils.getStrategyKey(strategyId));
    }

    /**
     * 获取广告位ID和名称缓存.
     *
     * @throws TuiaMediaException the tuia media exception
     */
    public boolean getSlotIdAndNameCache() throws TuiaMediaException {
        List<IdAndName> idAndNames = slotDAO.selectAllSlotIdAndName();
        for (IdAndName dto : idAndNames) {
            redisClient.setnx(CacheKeyUtils.getSlotIdAndNameKey(dto.getId()), dto.getName());
        }
        return true;
    }

    /**
     * delSlotIdAndNameCache:(删除广告位ID和名称缓存). <br/>
     *
     * @author ZFZ
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public boolean delSlotIdAndNameCache() throws TuiaMediaException {
        List<IdAndName> idAndNames = slotDAO.selectAllSlotIdAndName();
        for (IdAndName dto : idAndNames) {
            redisClient.del(CacheKeyUtils.getSlotIdAndNameKey(dto.getId()));
        }
        return true;
    }

    /**
     * updateSoltIdAndNameCache:(通过广告ID更新广告ID和广告名称缓存). <br/>
     *
     * @param slotId
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public boolean updateSoltIdAndNameCache(Long slotId) throws TuiaMediaException {
        String slotName= redisClient.get(CacheKeyUtils.getSlotIdAndNameKey(slotId));
        if (StringUtils.isBlank(slotName)) {
            SlotDto slotDto = slotDAO.selectById(slotId);
            if (slotDto!=null) {
                redisClient.setnx(CacheKeyUtils.getSlotIdAndNameKey(slotId), slotDto.getSlotName());
            }
         }
        return true;
    }
    
    /**
     * 
     * getSlotNameCacheById:(通过广告ID获取缓存的广告名称). <br/>
     *
     * @param slotId
     * @return
     * @since JDK 1.6
     */
    public String getSlotNameCacheById(Long slotId){
        return redisClient.get(CacheKeyUtils.getSlotIdAndNameKey(slotId));
    }
    
    
}
