/**
 * Project Name:media-biz File Name:DataStatisticsBOImpl.java Package Name:cn.com.duiba.tuia.media.bo.impl
 * Date:2016年10月12日下午5:20:36 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.bo.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.AccountDayReportDto;
import cn.com.duiba.tuia.media.api.dto.MeidaActivitySlotStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqSlotLineStatisticsByPageDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMediaAppDataStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotDataStatisticsDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotLineStatisticsDto;
import cn.com.duiba.tuia.media.bo.SlotLineStatisticsBO;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.DataTool;
import cn.com.duiba.tuia.media.common.utils.CollectionUtil;
import cn.com.duiba.tuia.media.dao.AccountDayReportDAO;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationDO;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import cn.com.duiba.tuia.media.service.SlotLineStatisticsService;
import cn.com.duiba.wolf.utils.DateUtils;

/**
 * ClassName:DataStatisticsBOImpl <br/>
 * Function: 结算数据. <br/>
 * Date: 2016年10月12日 下午5:20:36 <br/>
 * 
 * @author guyan
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class SlotLineStatisticsBOImpl implements SlotLineStatisticsBO {

    @Autowired
    SlotLineStatisticsService    slotLineStatisticsService;

    @Autowired
    SlotDAO                      slotDao;

    @Autowired
    MediaAppDao                  mediaAppDao;

    @Autowired
    AccountDayReportDAO          accountDayReportDAO;

    @Autowired
    MaterialSpecificationService materialSpecificationService;

    @Override
    public PageResultDto<RspMediaAppDataStatisticsDto> getMediaAppStatisticsData(ReqSlotLineStatisticsByPageDto param)
                                                                                                                      throws TuiaMediaException {
        int totalNum = this.slotLineStatisticsService.getMediaAppDataAmount(param);
        List<Long> mediaAppIds;
        List<MeidaActivitySlotStatisticsDto> mediaAppList;
        List<RspMediaAppDataStatisticsDto> rspMediaAppList = null;
        List<AccountDayReportDto> appReportList;
        if ((totalNum > 0) && (totalNum >= param.getRowStart())) {
            param.setPageSize(ReqSlotLineStatisticsByPageDto.PAGE_SIZE_50);
            param.setRowStart(param.getPageSize() * (param.getCurrentPage() - 1));
            mediaAppList = this.slotLineStatisticsService.getMediaAppStatisticsData(param);
            appReportList = accountDayReportDAO.selectAppDayBillList(param);//查询媒体日报表
            mediaAppIds = new ArrayList<>(mediaAppList.size());
            for (MeidaActivitySlotStatisticsDto mediaAppDto : mediaAppList) {
                mediaAppIds.add(mediaAppDto.getAppId());
            }
            Map<Long, String> mediaAppMap = getMediaAppMapByIds(mediaAppIds);
            Map<Long, Long> appConsumeMap = getAppConsume(appReportList);//获取媒体应用收益
            rspMediaAppList = rspMediaAppList(mediaAppList, mediaAppMap, appConsumeMap);
        }
        return new PageResultDto<>(totalNum, rspMediaAppList, param.getPageSize());
    }

    @Override
    public PageResultDto<RspSlotDataStatisticsDto> getSlotStatisticsData(ReqSlotLineStatisticsByPageDto param)
                                                                                                              throws TuiaMediaException {
        int totalNum = this.slotLineStatisticsService.getSlotDataAmount(param);
        List<Long> soltIds;
        List<MeidaActivitySlotStatisticsDto> slotStatsList;
        List<RspSlotDataStatisticsDto> rspSlotList = null;
        if ((totalNum > 0) && (totalNum >= param.getRowStart())) {
            param.setPageSize(ReqSlotLineStatisticsByPageDto.PAGE_SIZE_50);
            param.setRowStart(param.getPageSize() * (param.getCurrentPage() - 1));
            slotStatsList = this.slotLineStatisticsService.getSlotStatisticsData(param);
            soltIds = new ArrayList<>(slotStatsList.size());
            for (MeidaActivitySlotStatisticsDto slotDto : slotStatsList) {
                soltIds.add(slotDto.getSlotId());
            }
            Map<Long, String> soltMap = getSoltMapByIds(soltIds);
            rspSlotList = rspSlotList(slotStatsList, soltMap);
        }
        return new PageResultDto<>(totalNum, rspSlotList, param.getPageSize());
    }

    @Override
    public RspSlotLineStatisticsDto selectStatisticsDataByDate(ReqSlotLineStatisticsByPageDto param)
                                                                                                    throws TuiaMediaException {
        param.setOrder(ReqSlotLineStatisticsByPageDto.ORDER_TYPE);
        param.setSort(ReqSlotLineStatisticsByPageDto.SORT_TYPE);
        List<MeidaActivitySlotStatisticsDto> dataStatisticsList = this.slotLineStatisticsService.getStatisticsDataByDate(param);
        List<AccountDayReportDto> lineDataList=accountDayReportDAO.selectLineData(param);
        Map<String,Long> map=getLineDataConsume(lineDataList);
        return dataFormat(dataStatisticsList,map);
    }

    /**
     * 广告信息List转换成Map
     * 
     * @param soltIds
     * @return
     * @throws TuiaMediaException
     */
    private Map<Long, String> getSoltMapByIds(List<Long> soltIds) throws TuiaMediaException {
        List<IdAndName> list;
        Map<Long, String> slotMap = null;
        if (CollectionUtils.isNotEmpty(soltIds)) {
            slotMap = new HashMap<>(soltIds.size());
            list = slotDao.selectAppIdAndName(soltIds);
            for (IdAndName slotVO : list) {
                // key:广告位Id value:广告位名称
                slotMap.put(slotVO.getId(), slotVO.getName());
            }
        }
        return slotMap;
    }

    /**
     * 媒体app List转换成map
     * 
     * @param mediaAppIds
     * @return
     * @throws TuiaMediaException
     */
    private Map<Long, String> getMediaAppMapByIds(List<Long> mediaAppIds) throws TuiaMediaException {
        List<IdAndName> list;
        Map<Long, String> mediaAppMap = null;
        if (CollectionUtils.isNotEmpty(mediaAppIds)) {
            mediaAppMap = new HashMap<>(mediaAppIds.size());
            list = mediaAppDao.selectAppIdAndName(mediaAppIds);
            for (IdAndName mediaAppVO : list) {
                // key:媒体app Id value:媒体app 名称
                mediaAppMap.put(mediaAppVO.getId(), mediaAppVO.getName());
            }
        }
        return mediaAppMap;

    }

    /**
     * 拼装广告统计数据
     * 
     * @param slotList
     * @param slotMap
     * @return
     */
    private List<RspSlotDataStatisticsDto> rspSlotList(List<MeidaActivitySlotStatisticsDto> slotStatsList,
                                                       Map<Long, String> slotMap)
                                                                                                                   throws TuiaMediaException {
        List<RspSlotDataStatisticsDto> rspSoltList = new ArrayList<>(slotStatsList.size());
        Map<Long, MaterialSpecificationDO> msMap = materialSpecificationService.getMapByIds(CollectionUtil.getFieldList(slotStatsList,
                                                                                                                        "slotMsId"));
        for (MeidaActivitySlotStatisticsDto dto : slotStatsList) {
            RspSlotDataStatisticsDto rspDto = new RspSlotDataStatisticsDto();
            rspDto.setSlotId(dto.getSlotId());
            rspDto.setSlotType(dto.getSlotType());
            rspDto.setMsName(SlotDto.getSlotTypeNameBySlotType(dto.getSlotType()));
            // 获取自定义广告位规格名称
            if (dto.getSlotMsId() != null) {
                MaterialSpecificationDO ms = msMap.get(dto.getSlotMsId());
                if (ms != null) {
                    rspDto.setMsName(ms.getTitle());
                }
            }
            rspDto.setClickCount(dto.getActClickCount());
            rspDto.setExposureCount(dto.getActExposeCount());
            rspDto.setSlotName(slotMap.get(dto.getSlotId()));
            // 数据计算
            calculateSlotStatisticsData(rspDto);
            rspSoltList.add(rspDto);
        }
        return rspSoltList;
    }

    /**
     * 拼装媒体app统计数据
     * 
     * @param mediaAppList
     * @param mediaAppMap
     * @return
     */
    private List<RspMediaAppDataStatisticsDto> rspMediaAppList(List<MeidaActivitySlotStatisticsDto> mediaAppList,
                                                               Map<Long, String> mediaAppMap,
                                                               Map<Long, Long> appConsumeMap) {
        List<RspMediaAppDataStatisticsDto> rspSoltList = new ArrayList<>(mediaAppList.size());
        for (MeidaActivitySlotStatisticsDto dto : mediaAppList) {
            RspMediaAppDataStatisticsDto rspDto = new RspMediaAppDataStatisticsDto();
            rspDto.setAppId(dto.getAppId());
            rspDto.setAppName(mediaAppMap.get(dto.getAppId()));
            rspDto.setClickCount(dto.getActClickCount());
            rspDto.setExposureCount(dto.getActExposeCount());
            rspDto.setPlatform(dto.getPlatform());
            rspDto.setConsumeTotal(appConsumeMap.get(dto.getAppId()));
            // 数据计算
            calculateMediaAppStatisticsData(rspDto);
            rspSoltList.add(rspDto);
        }
        return rspSoltList;
    }

    /**
     * 计算媒体app统计数据.
     *
     * @param data the data
     */
    private void calculateMediaAppStatisticsData(RspMediaAppDataStatisticsDto data) {
        // 计算点击率 =点击量/曝光量 *100
        if (data.getExposureCount() != null) {
            data.setClickRate(calculateClickRade(data.getClickCount(), data.getExposureCount()));
        }
    }

    /**
     * 计算广告位统计数据.
     *
     * @param data the data
     */
    private void calculateSlotStatisticsData(RspSlotDataStatisticsDto data) {
        // 计算点击率 =点击量/曝光量 *100
        if (data.getExposureCount() != null) {
            data.setClickRate(calculateClickRade(data.getClickCount(), data.getExposureCount()));
        }
    }

    /**
     * 拼装成折线图的数据格式
     * 
     * @param dataList
     * @return
     */
    private RspSlotLineStatisticsDto dataFormat(List<MeidaActivitySlotStatisticsDto> dataList,Map<String,Long> lineDataMap) {
        List<String> curDateList;
        List<Long> exposureCountList;
        List<Long> clickCountList;
        List<Long> consumeTotalList;
        List<Long> ecpmList;
        List<Float> clickRateList;
        RspSlotLineStatisticsDto rspDto = new RspSlotLineStatisticsDto();
        if (CollectionUtils.isNotEmpty(dataList)) {
            curDateList = new ArrayList<>(dataList.size());
            exposureCountList = new ArrayList<>(dataList.size());
            clickCountList = new ArrayList<>(dataList.size());
            consumeTotalList = new ArrayList<>(dataList.size());
            ecpmList = new ArrayList<>(dataList.size());
            clickRateList = new ArrayList<>(dataList.size());
            for (MeidaActivitySlotStatisticsDto dto : dataList) {
                String dayStr=DateUtils.getDayStr(DateUtils.getDayDate(dto.getCurDate()));
                curDateList.add(dayStr);// 时间格式化 "yyyy-MM-dd"
                clickCountList.add(dto.getActClickCount());
                exposureCountList.add(dto.getActExposeCount());
                ecpmList.add(calculateEcpmData(dto));
                consumeTotalList.add(lineDataMap.get(dayStr)); //获取预计收入
                clickRateList.add(calculateClickRateData(dto));
            }
            rspDto.setClickCount(clickCountList);
            rspDto.setConsumeTotal(consumeTotalList);
            rspDto.setCurDate(curDateList);
            rspDto.seteCpm(ecpmList);
            rspDto.setExposureCount(exposureCountList);
            rspDto.setClickRate(clickRateList);
        }
        return rspDto;
    }

    /**
     * 计算ecpm统计数据.
     *
     * @param data the data
     */
    private Long calculateEcpmData(MeidaActivitySlotStatisticsDto data) {
        // eCPM=(预计收入/曝光量*1000)
        return DataTool.calculateEcpm(data.getConsumeTotal(), data.getActExposeCount());
    }

    /**
     * 计算点击率 点击量/曝光量 *100
     * 
     * @param data
     * @return
     */
    private Float calculateClickRateData(MeidaActivitySlotStatisticsDto data) {
        // clickRate=(点击量/曝光量 *100)
        return calculateClickRade(data.getActClickCount(), data.getActExposeCount());
    }

    /**
     * 计算点击率 点击量/曝光量 *100
     * 
     * @param divisor
     * @param dividend
     * @return
     */
    private static Float calculateClickRade(Long divisor, Long dividend) {
        Float result = 0F;
        if (dividend != null && dividend.longValue() != 0 && divisor != null) {
            BigDecimal bigDivisor = new BigDecimal(divisor * 100);
            BigDecimal bigDividend = new BigDecimal(dividend);
            return bigDivisor.divide(bigDividend, 2, BigDecimal.ROUND_HALF_UP).floatValue();
        }
        return result;
    }

    /**
     * getAppConsume:(查询媒体应用预计收益). <br/>
     *
     * @param appReportList
     * @return
     * @since JDK 1.6
     */
    private Map<Long, Long> getAppConsume(List<AccountDayReportDto> appReportList) {
        HashMap<Long, Long> map = new HashMap<>();
        for (AccountDayReportDto dto : appReportList) {
            map.put(dto.getAppId(), dto.getConsumeTotal());
        }
        return map;
    }

    /**
     * getSlotConsume:(查询折线图查询每天的总预计收益). <br/>
     *
     * @param slotReportList
     * @return
     * @since JDK 1.6
     */
    private Map<String, Long> getLineDataConsume(List<AccountDayReportDto> lineDataList) {
        HashMap<String, Long> map = new HashMap<>();
        for (AccountDayReportDto dto : lineDataList) {
            map.put(DateUtils.getDayStr(DateUtils.getDayDate(dto.getCurDate())), dto.getConsumeTotal());
        }
        return map;
    }
}
