package cn.com.duiba.tuia.media.service.cacheservice;

import java.util.ArrayList;
import java.util.List;

import cn.com.duiba.tuia.media.api.constant.SplitConstant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import com.google.common.base.Joiner;

import cn.com.duiba.tuia.media.api.utils.CacheKeyUtils;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.wolf.redis.RedisClient;

/**
 * @author xuyenan
 * @createTime 2016/12/28
 */
@Service
public class ActivityAdvertCacheService {

    @Autowired
    private RedisClient redisClient;

    /**
     * 从缓存中获取活动关联广告
     * 
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @return 广告ID列表
     */
    public List<Long> get(Long activityId, Integer activityType) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivityAdvertByKey(activityId, activityType);
        String value = redisClient.get(key);
        if (StringUtils.isEmpty(value)) {
            return new ArrayList<>();
        }
        List<Long> advertIds = new ArrayList<>();
        for (String id : value.split(SplitConstant.SPLIT_COMMA)) {
            advertIds.add(Long.parseLong(id));
        }
        return advertIds;
    }

    /**
     * 活动关联广告写入缓存
     * 
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @param advertIds 广告券列表
     */
    public void set(Long activityId, Integer activityType, List<Long> advertIds) {
        String key = CacheKeyUtils.getActivityAdvertByKey(activityId, activityType);
        String value = Joiner.on(SplitConstant.SPLIT_COMMA).join(advertIds);
        redisClient.set(key, value);
    }
    
    /**
     * 活动关联广告写入缓存
     * 
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @param advertIds 广告券列表
     */
    public void setBySource(Long activityId, Integer source, List<Long> advertIds) {
        String key = CacheKeyUtils.getActivityAdvertByKeyAndSource(activityId, source);
        String value = Joiner.on(SplitConstant.SPLIT_COMMA).join(advertIds);
        redisClient.set(key, value);
    }
    
    /**
     * 从缓存中获取活动关联广告
     * 
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @return 广告ID列表
     */
    public List<Long> getBySource(Long activityId, Integer source) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivityAdvertByKeyAndSource(activityId, source);
        String value = redisClient.get(key);
        if (StringUtils.isEmpty(value)) {
            return new ArrayList<>();
        }
        List<Long> advertIds = new ArrayList<>();
        for (String id : value.split(SplitConstant.SPLIT_COMMA)) {
            advertIds.add(Long.parseLong(id));
        }
        return advertIds;
    }
}
