package cn.com.duiba.tuia.media.service.cacheservice;

import cn.com.duiba.tuia.media.api.constant.SplitConstant;
import cn.com.duiba.tuia.media.api.utils.CacheKeyUtils;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.MediaAppActivityDAO;
import cn.com.duiba.tuia.media.dataobject.MediaAppActivityDO;
import cn.com.duiba.wolf.redis.RedisClient;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author xuyenan
 * @createTime 2017/3/20
 */
@Service
public class ActivityDirectAppCacheService {

    @Autowired
    private RedisClient         redisClient;
    @Autowired
    private MediaAppActivityDAO mediaAppActivityDAO;

    private static final String NO_APP = "NO_APP";

    /**
     * 从缓存获取活动定向媒体
     *
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @return 活动
     */
    public List<Long> get(Long activityId, Integer activityType) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivity_direct_app_key(activityId, activityType);
        String value = redisClient.get(key);
        if (NO_APP.equals(value)) {
            return Collections.emptyList();
        }
        if (StringUtils.isEmpty(value)) {
            List<Long> appList = mediaAppActivityDAO.getMediaAppByActivity(activityId, activityType);
            set(activityId, activityType, appList);
            return appList;
        }
        List<Long> appList = new ArrayList<>();
        for (String appId : value.split(SplitConstant.SPLIT_COMMA)) {
            appList.add(Long.parseLong(appId));
        }
        return appList;
    }
    
    /**
     * 从缓存获取活动定向媒体
     *
     * @param activityId 活动ID
     * @param source 活动来源
     * @return 活动
     */
    public List<Long> getBySource(Long activityId, Integer source) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivityDirectAppKeyBySource(activityId, source);
        String value = redisClient.get(key);
        if (NO_APP.equals(value)) {
            return Collections.emptyList();
        }
        if (StringUtils.isEmpty(value)) {
            List<Long> appList = mediaAppActivityDAO.getMediaAppByActivityAndSource(activityId, source);
            //TODO
            //set(activityId, source, appList);
            setBySource(activityId, source, appList);
            return appList;
        }
        List<Long> appList = new ArrayList<>();
        for (String appId : value.split(SplitConstant.SPLIT_COMMA)) {
            appList.add(Long.parseLong(appId));
        }
        return appList;
    }

    /**
     * 从缓存获取活动定向媒体
     *
     * @param activityIds 活动ID
     * @param activityTypes 活动类型
     * @return 活动
     * @throws TuiaMediaException
     */
    public List<MediaAppActivityDO> get4List(List<Long> activityIds, List<Integer> activityTypes)
                                                                                                 throws TuiaMediaException {
        List<MediaAppActivityDO> mediaAppActivityDOs = Lists.newArrayList();
        if (CollectionUtils.isEmpty(activityIds) || CollectionUtils.isEmpty(activityTypes)) {
            return mediaAppActivityDOs;
        } else {
            mediaAppActivityDOs = mediaAppActivityDAO.getMediaAppByActivity4List(activityIds, activityTypes);
            return mediaAppActivityDOs;
        }

    }
    

    /**
     * 从缓存获取活动定向媒体(通过来源)
     *
     * @param activityIds 活动ID
     * @param sources 活动类型
     * @return 活动
     * @throws TuiaMediaException
     */
    public List<MediaAppActivityDO> get4ListBySources(List<Long> activityIds, List<Integer> sources)
                                                                                                 throws TuiaMediaException {
        List<MediaAppActivityDO> mediaAppActivityDOs = Lists.newArrayList();
        if (CollectionUtils.isEmpty(activityIds) || CollectionUtils.isEmpty(sources)) {
            return mediaAppActivityDOs;
        } else {
            //TODO
            mediaAppActivityDOs = mediaAppActivityDAO.getMediaAppByActivity4ListBySource(activityIds, sources);
            return mediaAppActivityDOs;
        }

    }

    /**
     * 写入缓存
     *
     * @param activityId
     * @param activityType
     * @param appList
     */
    public void set(Long activityId, Integer activityType, List<Long> appList) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivity_direct_app_key(activityId, activityType);
        // 如果没有定向列表，塞特殊值
        if (CollectionUtils.isEmpty(appList)) {
            redisClient.setex(key, 3600 * 24, NO_APP);
            return;
        }
        StringBuilder sb = new StringBuilder();
        String value = Joiner.on(SplitConstant.SPLIT_COMMA).appendTo(sb, appList).toString();
        redisClient.setex(key, 3600 * 24, value);
    }
    
    /**
     * 写入缓存
     *
     * @param activityId
     * @param activityType
     * @param appList
     */
    public void setBySource(Long activityId, Integer source, List<Long> appList) throws TuiaMediaException {
        String key = CacheKeyUtils.getActivityDirectAppKeyBySource(activityId, source);
        // 如果没有定向列表，塞特殊值
        if (CollectionUtils.isEmpty(appList)) {
            redisClient.setex(key, 3600 * 24, NO_APP);
            return;
        }
        StringBuilder sb = new StringBuilder();
        String value = Joiner.on(SplitConstant.SPLIT_COMMA).appendTo(sb, appList).toString();
        redisClient.setex(key, 3600 * 24, value);
    }
}
