/**
 * Project Name:media-biz<br>
 * File Name:AccountServiceImpl.java<br>
 * Package Name:cn.com.duiba.tuia.media.service.impl<br>
 * Date:2016年9月26日上午10:52:36<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */
package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.media.api.dto.AccountChangeDto;
import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.api.dto.MediaCompanyMsgDto;
import cn.com.duiba.tuia.media.api.dto.MediaPersonalMsgDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqGetAccountByPageDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAccountBaseDto;
import cn.com.duiba.tuia.media.api.utils.CacheKeyUtils;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.constants.TuiaDomainConstant;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.IDGeneratorTool;
import cn.com.duiba.tuia.media.common.utils.BlowfishUtils;
import cn.com.duiba.tuia.media.common.utils.MessageDigestUtils;
import cn.com.duiba.tuia.media.dao.AccountBankDAO;
import cn.com.duiba.tuia.media.dao.AccountCheckRecordDAO;
import cn.com.duiba.tuia.media.dao.AccountDAO;
import cn.com.duiba.tuia.media.dao.MediaCompanyMsgDAO;
import cn.com.duiba.tuia.media.dao.MediaPersonalMsgDAO;
import cn.com.duiba.tuia.media.domain.AccountBankDto;
import cn.com.duiba.tuia.media.model.Email;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.model.req.BaseBankInfo;
import cn.com.duiba.tuia.media.model.req.ReqSendSMS;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAccountBase;
import cn.com.duiba.tuia.media.model.req.ReqUpdateCompanyInfo;
import cn.com.duiba.tuia.media.model.req.ReqUpdatePersonInfo;
import cn.com.duiba.tuia.media.model.rsp.TokenRsp;
import cn.com.duiba.tuia.media.service.AccountChangeService;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.service.EmailService;
import cn.com.duiba.tuia.media.service.SmsService;
import cn.com.duiba.tuia.media.utils.CheckAccountVaildStatus;

/**
 * ClassName: AccountServiceImpl <br/>
 * date: 2016年9月26日 上午10:52:36 <br/>
 *
 * @author xiawei
 * @version
 * @since JDK 1.6
 */
@Service
public class AccountServiceImpl extends BaseCacheService implements AccountService {

    /** 邮箱验证码位数. */
    private static final int        EMAIL_CODE_LENGTH = 6;

    /** 10分钟. */
    private static final int        THIRTY_MINUTES    = 3 * 10 * 60;

    /** 48小时. */
    private static final int        FORTYEIGHT_HOURS  = 48 * 60 * 60;

    /** The login encrypt key. */
    @Value("${media.login.encrypt.key}")
    private String                  loginEncryptKey;

    /** The email verify key. */
    @Value("${media.email.verify.key}")
    private String                  emailVerifyKey;

    /** The email service. */
    @Autowired
    private EmailService            emailService;

    @Autowired
    private AccountDAO              accountDAO;

    @Autowired
    private AccountCheckRecordDAO   accountCheckRecordDao;

    @Autowired
    private MediaCacheService       mediaCacheService;

    @Autowired
    private AccountChangeService    accountChangeService;

    @Autowired
    private AccountBankDAO          accountBankDAO;

    @Autowired
    private MediaCompanyMsgDAO      mediaCompanyMsgDAO;

    @Autowired
    private MediaPersonalMsgDAO     mediaPersonalMsgDAO;

    @Autowired
    private SmsService              smsService;

    @Autowired
    private CheckAccountVaildStatus checkAccountVaildStatus;

    private String                  linkmanStr        = "linkman";

    private String                  emailStr          = "email";

    @Override
    public boolean isEmailExists(String email) throws TuiaMediaException {
        return accountDAO.selectByEmail(email) != null;
    }

    @Override
    public boolean sendResetPasswdEmail(final String email) throws TuiaMediaException {

        // 查询账号信息
        final AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        // 发送邮件
        doSendResetPasswdEmail(accountDto.getLinkman(), email);

        return true;

    }

    @Override
    public boolean sendReplaceEmail(String email, Long userId) throws TuiaMediaException {

        AccountDto user = selectByIdNotNull(userId);
        // 校验该邮箱是否被使用
        AccountDto accountDto = accountDAO.selectByEmail(email);
        if (accountDto != null) {
            if (accountDto.getMediaId().longValue() == userId) {
                // 邮箱没有修改
                throw new TuiaMediaException(ErrorCode.E0104017);
            } else {
                // 邮箱被占用
                throw new TuiaMediaException(ErrorCode.E0102016);
            }
        }

        doSendReplaceEmail(user.getLinkman(), email, userId);

        return true;
    }

    @Override
    public boolean replaceEmail(String token, Long userId) throws TuiaMediaException {
        String cacheEmail = redisClient.get(CacheKeyUtils.getReplaceEmailKey(userId, token));
        if (StringUtils.isEmpty(cacheEmail)) {
            // 验证码失效
            throw new TuiaMediaException(ErrorCode.E0102003);
        }
        if (accountDAO.updateEmail(userId, cacheEmail) != 1) {
            // 更换邮箱失败
            throw new TuiaMediaException(ErrorCode.E0102018);
        }
        redisClient.del(CacheKeyUtils.getReplaceEmailKey(userId, token));
        return true;
    }

    @Override
    public AccountDto selectByIdNotNull(Long mediaId) throws TuiaMediaException {
        AccountDto user = accountDAO.selectById(mediaId);
        if (user == null) {
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

        return user;
    }

    @Override
    public TokenRsp resetPasswdRedirect(String email, String verificationCode) throws TuiaMediaException {
        String cacheCode = redisClient.get(CacheKeyUtils.getResetPwEmailKey(email));
        if (!StringUtils.equals(verificationCode, cacheCode)) {
            // 验证码失效
            logger.error("the verificationCode invalid, the email=[{}]", email);
            throw new TuiaMediaException(ErrorCode.E0102003);
        }
        String key = new Date().getTime() + "-" + email;
        String token = BlowfishUtils.encryptBlowfish(key, emailVerifyKey);
        redisClient.setex(token, THIRTY_MINUTES, email);
        TokenRsp tokenRsp = new TokenRsp();
        tokenRsp.setEmail(email);
        tokenRsp.setToken(token);
        return tokenRsp;
    }

    @Override
    public boolean resetPassword(String email, String newPasswd, String token) throws TuiaMediaException {
        // 校验重置密码的toekn
        String cacheToken = redisClient.get(token);
        if (null == cacheToken) {
            // token失效
            logger.error("the token invalid, the email=[{}]", email);
            throw new TuiaMediaException(ErrorCode.E0102009);
        }

        String newPd = BlowfishUtils.encryptBlowfish(MessageDigestUtils.sha(newPasswd), loginEncryptKey);
        // 查询账号
        AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        if (newPd.equals(accountDto.getPassword())) {
            // 新密码和旧密码不能一致
            throw new TuiaMediaException(ErrorCode.E0102013);
        }

        int count = accountDAO.updatePasswdByEmail(email, newPd);

        if (count > 0) {
            // 重置成功， 删除缓存验证码,token
            redisClient.del(CacheKeyUtils.getResetPwEmailKey(email));
            redisClient.del(token);
            return true;
        }
        return false;
    }

    @Override
    public void verifyEmail(String token) throws TuiaMediaException {
        // 从缓存里查询该token
        String value = redisClient.get(token);
        if (value == null) {
            logger.error("token has invalid, the token=[{}]", token);
            throw new TuiaMediaException(ErrorCode.E0102003);
        }

        // 查询账号
        AccountDto accountDto = accountDAO.selectByEmail(value);
        accountDtoIsNull(accountDto);
        // 更新邮箱审核状态
        accountDAO.updateEmailStatus(accountDto.getMediaId(), AccountDto.EMAIL_STATUS_CHECKED);

        redisClient.del(token);
    }

    /**
     * 构建和发送重置密码邮件.
     *
     * @param userId 用户ID
     * @param linkman 用户名称
     * @param userEmail the user email
     */
    private void doSendResetPasswdEmail(final String linkman, final String userEmail) {
        // 发送邮件
        executorService.submit(new Runnable() {//NOSONAR

            @Override
            public void run() {
                logger.info("account is reset,the email is sending =[{}]",userEmail);
                // 生成验证码
                String verificationCode = IDGeneratorTool.getRandomString(EMAIL_CODE_LENGTH);

                Map<String, Object> model = new HashMap<>(3);
                model.put(linkmanStr, linkman);
                model.put(emailStr, userEmail);
                model.put("verificationCode", verificationCode);

                // 拼装邮件内容
                Email email = new Email();
                email.setTo(userEmail);
                email.setTemplateLocation("/templates/mail/resetpasswd.vm");
                email.setSubject("推啊媒体账号找回密码-邮箱验证码");
                email.setModel(model);
                
                logger.info("account is reset,the email is sending =[{}]",userEmail);
                
                emailService.sendWithTemplate(email);
                logger.info("the email is sended =[{}]",userEmail);
                // 存入缓存
                redisClient.setex(CacheKeyUtils.getResetPwEmailKey(userEmail), THIRTY_MINUTES, verificationCode);
            }
        });
    }

    /**
     * 发生更换邮箱的邮件.
     *
     * @param linkman the linkman
     * @param userEmail the user email
     */
    private void doSendReplaceEmail(final String linkman, final String userEmail, final Long userId) {
        // 发送邮件
        executorService.submit(new Runnable() {//NOSONAR

            @Override
            public void run() {
                logger.info("account is replace,the email is sending =[{}]",userEmail);
                // 生成验证码
                String verificationCode = IDGeneratorTool.getRandomString(EMAIL_CODE_LENGTH);

                Map<String, Object> model = new HashMap<>(3);
                model.put(linkmanStr, linkman);
                model.put(emailStr, userEmail);
                model.put("verificationCode", verificationCode);

                // 拼装邮件内容
                Email email = new Email();
                email.setTo(userEmail);
                email.setTemplateLocation("/templates/mail/replaceEmail.vm");
                email.setSubject("推啊媒体更换邮箱通知");
                email.setModel(model);
                
                logger.info("account is replace,the email is sending =[{}]",userEmail);
                
                emailService.sendWithTemplate(email);

                logger.info("send email is param the linkMan=[{}],the email=[{}],the verificationCode=[{}]", linkman, email, verificationCode);
                // 存入缓存
                redisClient.setex(CacheKeyUtils.getReplaceEmailKey(userId, verificationCode), ONE_HOUR, userEmail);
            }
        });
    }

    /**
     * Do send check email.
     *
     * @param accountDO the account do
     */
    @Override
    public void doSendVerifyEmail(final AccountDto accountDto) {
        executorService.submit(new Runnable() {//NOSONAR

            @Override
            public void run() {
                
                logger.info("account is register,the email is sending =[{}]",accountDto.getEmail());
                
                String key = new Date().getTime() + "-" + accountDto.getMediaId();
                String code = BlowfishUtils.encryptBlowfish(key, emailVerifyKey);
                String email = accountDto.getEmail();
                String url = TuiaDomainConstant.getTuiaDomainName() + "/account/verifyEmail?email=" + email + "&token=" + code;
                Map<String, Object> model = new HashMap<>();
                model.put(linkmanStr, accountDto.getLinkman());
                model.put(emailStr, email);
                model.put("url", url);

                // 拼装邮件内容
                Email sendEmail = new Email();
                sendEmail.setTo(accountDto.getEmail());
                sendEmail.setTemplateLocation("/templates/mail/active.vm");
                sendEmail.setSubject("推啊媒体账号邮箱验证");
                sendEmail.setModel(model);
                
                logger.info("account is register,the email is sending =[{}]",accountDto.getEmail());
                
                emailService.sendWithTemplate(sendEmail);
                
                logger.info("email is send,the email =[{}]",sendEmail.getTo());
                
                redisClient.setex(code, FORTYEIGHT_HOURS, accountDto.getEmail());
            }
        });
    }

    @Override
    public void repeatVerifyEmail(String email) throws TuiaMediaException {
        AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        int emailStatus = accountDto.getEmailStatus();
        if (AccountDto.EMAIL_STATUS_CHECKED == emailStatus) {
            throw new TuiaMediaException(ErrorCode.E0102012);
        }
        // 重新发送验证邮件
        doSendVerifyEmail(accountDto);
    }

    @Override
    public boolean updateAccountBase(ReqUpdateAccountBase req) throws TuiaMediaException {
        return accountDAO.updateAccount(req) == 1;
    }

    @Override
    public PageResultDto<RspAccountBaseDto> pageQuery(ReqGetAccountByPageDto req) throws TuiaMediaException {
        // 查询总数
        Integer totalAmount = accountDAO.selectMediaDataAmount(req);
        List<AccountDto> accountDtos = null;
        // 总数查询有结果，查询分页数据
        List<RspAccountBaseDto> mediaRsps = new ArrayList<>();
        if ((totalAmount != 0) && (totalAmount >= req.getRowStart())) {
            req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
            accountDtos = accountDAO.selectMediaDataByPage(req);
        }

        if (CollectionUtils.isNotEmpty(accountDtos)) {
            HashMap<Long, Long> blanceMap = buildBlanceData(accountDtos); // 账户余额
            for (AccountDto accountDto : accountDtos) {
                RspAccountBaseDto rspAccountBase = new RspAccountBaseDto();
                Long mediaId = accountDto.getMediaId();
                rspAccountBase.setMediaId(mediaId);
                rspAccountBase.setEmail(accountDto.getEmail());
                rspAccountBase.setEmailStatus(accountDto.getEmailStatus());
                rspAccountBase.setCompanyName(accountDto.getCompanyName());
                rspAccountBase.setBusinessLicenseId(accountDto.getBusinessLicenseId());
                rspAccountBase.setBusinessLicenseUrl(accountDto.getBusinessLicenseUrl());
                rspAccountBase.setLinkman(accountDto.getLinkman());
                rspAccountBase.setLinkmanPhone(accountDto.getLinkPhone());
                rspAccountBase.setCheckStatus(accountDto.getCheckStatus());
                rspAccountBase.setFreezeStatus(accountDto.getFreezeStatus());
                rspAccountBase.setBalance(blanceMap.get(mediaId) == null ? AccountChangeDto.AMOUNT_INIT_VAL : blanceMap.get(mediaId));
                mediaRsps.add(rspAccountBase);
            }
        }

        return new PageResultDto<>(totalAmount, mediaRsps, req.getPageSize());
    }

    @Override
    public boolean updateFreezeStatus(Long mediaId, Integer status) throws TuiaMediaException {
        boolean flag = accountDAO.updateFreezeStatus(mediaId, status) == 1 ? true : false;
        if (flag) {
            mediaCacheService.updateMediaCache(mediaId, checkAccountVaildStatus.doGetAccountVaildStatus(mediaId));
        }
        return flag;
    }

    @Override
    public boolean insertAccountBase(AccountDto accountDto) throws TuiaMediaException {
        return accountDAO.insert(accountDto) == 1 ? true : false;
    }

    @Override
    public AccountDto selectByMediaId(Long mediaId) throws TuiaMediaException {
        return accountDAO.selectById(mediaId);
    }

    @Override
    public AccountDto selectByEmail(String email) throws TuiaMediaException {
        return accountDAO.selectByEmail(email);
    }

    @Override
    public boolean updateAuditData(AccountDto req) throws TuiaMediaException {
        return accountDAO.updateAuditData(req) == 1 ? true : false;
    }

    @Override
    public void accountDtoIsNull(AccountDto dto) throws TuiaMediaException {
        if (dto == null) {
            logger.error("the account is not exist");
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

    }

    @Override
    public AccountDto isPhoneExists(String linkPhone) throws TuiaMediaException {
        return accountDAO.selectByPhone(linkPhone);
    }

    @Override
    public boolean repeatRegisterVerifyPhone(String linkPhone, Long mediaId) throws TuiaMediaException {
        AccountDto targetAccount = isPhoneExists(linkPhone);
        if (mediaId != null) {
            // 表示重新提交注册信息时验证手机号
            AccountDto sourceAccount = accountDAO.selectById(mediaId);
            // 判断账户是否存在
            accountDtoIsNull(sourceAccount);
            if (targetAccount != null && !sourceAccount.getLinkPhone().equals(targetAccount.getLinkPhone())) {
                return true;
            }
            return false;
        }
        // 表示注册信息时验证手机号
        if (targetAccount != null) {
            return true;
        }
        return false;
    }

    /**
     * bildBlanceData:(构造账户余额数据). <br/>
     *
     * @param accountList
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private HashMap<Long, Long> buildBlanceData(List<AccountDto> accountList) throws TuiaMediaException {
        List<Long> mediaIds = new ArrayList<>(accountList.size());
        for (AccountDto accountDto : accountList) {
            mediaIds.add(accountDto.getMediaId());
        }
        List<AccountChangeDto> accountChangeList = accountChangeService.selectAccountChangeListByIds(mediaIds);
        HashMap<Long, Long> map = new HashMap<>(accountChangeList.size());
        for (AccountChangeDto accountChangeDto : accountChangeList) {
            map.put(accountChangeDto.getMediaId(), accountChangeDto.getBalance());
        }
        return map;
    }

    @Override
    public void updateCompanyInfo(ReqUpdateCompanyInfo req) throws TuiaMediaException {
        Date today = new Date();
        Long mediaId = req.getMediaId();

        logger.info(" updateCompanyInfo the req = [{}]", req.toString());

        AccountDto accountDto = accountDAO.selectById(mediaId);
        if (accountDto == null) {
            logger.error("the account is not exist");
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

        if (AccountDto.ROLE_TYPE_DEFAULT != accountDto.getRoleType() && AccountDto.ROLE_TYPE_COMPANY != accountDto.getRoleType()) {
            logger.error("the account roleType is not company");
            throw new TuiaMediaException(ErrorCode.E0102024);
        }

        MediaCompanyMsgDto dto = mediaCompanyMsgDAO.selectCompanyMsgById(mediaId);
        logger.info(" updateCompanyInfo get MediaCompanyMsgDto = [{}]", dto);
        if (dto == null) {
            dto = new MediaCompanyMsgDto();
            doSetMediaCompanyMsgDto(req, dto, today, mediaId);
            mediaCompanyMsgDAO.insertCompanyMsg(dto);
        } else {
            doSetMediaCompanyMsgDto(req, dto, today, mediaId);
            mediaCompanyMsgDAO.updateCompanyMsg(dto);
        }

        doUpdateAccountBankInfo(req, today, mediaId);

        if (AccountDto.ROLE_TYPE_DEFAULT == accountDto.getRoleType()) {
            accountDAO.updateRoleType(AccountDto.ROLE_TYPE_COMPANY, mediaId);
        }
    }

    private void doSetMediaCompanyMsgDto(ReqUpdateCompanyInfo req, MediaCompanyMsgDto dto, Date today, Long mediaId) {
        dto.setBusinessLicenseId(req.getBusinessLicenseId());
        dto.setBusinessLicenseUrl(req.getBusinessLicenseUrl());
        dto.setFinanceCompanyName(req.getFinanceCompanyName());
        dto.setGmtCreate(today);
        dto.setGmtModified(today);
        dto.setMediaId(mediaId);
    }

    private void doUpdateAccountBankInfo(BaseBankInfo req, Date today, Long mediaId) throws TuiaMediaException {
        logger.info(" doUpdateAccountBankInfo the req = [{}]", req.toString());
        AccountBankDto accountBankDto = accountBankDAO.selectById(mediaId);
        logger.info(" doUpdateAccountBankInfo the accountBankDto = [{}]", accountBankDto);
        if (accountBankDto == null) {
            accountBankDto = new AccountBankDto();
            doSetAccountBankDto(req, today, mediaId, accountBankDto);
            accountBankDAO.insert(accountBankDto);
        } else {
            doSetAccountBankDto(req, today, mediaId, accountBankDto);
            accountBankDAO.update(accountBankDto);
        }
    }

    private void doSetAccountBankDto(BaseBankInfo req, Date today, Long mediaId, AccountBankDto accountBankDto) {
        accountBankDto.setBankName(req.getBankName());
        accountBankDto.setBranchName(req.getBranchName());
        accountBankDto.setCardNumber(req.getCardNumber());
        accountBankDto.setCity(req.getCity());
        accountBankDto.setGmtCreate(today);
        accountBankDto.setGmtModified(today);
        accountBankDto.setMediaId(mediaId);
        accountBankDto.setProvince(req.getProvince());
    }

    @Override
    public void updatePersonInfo(ReqUpdatePersonInfo req) throws TuiaMediaException {
        Date today = new Date();
        Long mediaId = req.getMediaId();

        logger.info(" updatePersonInfo the req = [{}]", req.toString());

        AccountDto accountDto = accountDAO.selectById(mediaId);
        if (accountDto == null) {
            logger.error("the account is not exist");
            throw new TuiaMediaException(ErrorCode.E0102001);
        }

        if (AccountDto.ROLE_TYPE_DEFAULT != accountDto.getRoleType() && AccountDto.ROLE_TYPE_PERSONAL != accountDto.getRoleType()) {
            logger.error("the account roleType is not personal");
            throw new TuiaMediaException(ErrorCode.E0102024);
        }

        MediaPersonalMsgDto dto = mediaPersonalMsgDAO.selectPersonalMsgById(mediaId);
        logger.info(" updatePersonInfo get MediaCompanyMsgDto = [{}]", dto);
        if (dto == null) {
            dto = new MediaPersonalMsgDto();
            doSetMediaPersonalMsgDto(req, today, mediaId, dto);
            mediaPersonalMsgDAO.insertPersonalMsg(dto);
        } else {
            doSetMediaPersonalMsgDto(req, today, mediaId, dto);
            mediaPersonalMsgDAO.updatePersonalMsg(dto);
        }
        doUpdateAccountBankInfo(req, today, mediaId);

        if (AccountDto.ROLE_TYPE_DEFAULT == accountDto.getRoleType()) {
            accountDAO.updateRoleType(AccountDto.ROLE_TYPE_PERSONAL, mediaId);
        }
    }

    private void doSetMediaPersonalMsgDto(ReqUpdatePersonInfo req, Date today, Long mediaId, MediaPersonalMsgDto dto) {
        dto.setGmtCreate(today);
        dto.setGmtModified(today);
        dto.setIdCard(req.getIdCard());
        dto.setIdCardBackUrl(req.getIdCardBackUrl());
        dto.setIdCardFrontUrl(req.getIdCardFrontUrl());
        dto.setMediaId(mediaId);
        dto.setPersonalName(req.getPersonalName());
    }

    @Override
    public boolean sendReplacePhone(ReqSendSMS req) throws TuiaMediaException {
        Long mediaId = req.getMediaId();
        AccountDto dto = accountDAO.selectById(mediaId);
        if (dto == null) {
            throw new TuiaMediaException(ErrorCode.E0102001);
        }
        AccountDto accountDto = accountDAO.selectByPhone(req.getLinkPhone());
        if (accountDto != null) {
            if (accountDto.getMediaId().longValue() == mediaId) {
                // 手机号没有修改
                throw new TuiaMediaException(ErrorCode.E0102023);
            } else {
                // 手机号已经被注册
                throw new TuiaMediaException(ErrorCode.E0102021);
            }
        }
        smsService.send(req); // 发送更换短信息
        return true;
    }

    @Override
    public boolean updatePhoneNum(Long mediaId, String phone) throws TuiaMediaException {
        return accountDAO.updatePhoneNum(mediaId, phone) == 1;
    }

    @Override
    public List<IdAndName> selectIdAndNameByIds(List<Long> mediaIds) throws TuiaMediaException {
        return accountDAO.selectMediaIdAndEmails(mediaIds);
    }

    @Override
    public boolean manualUpdateEmailStatus(String email) throws TuiaMediaException {
        // 查询账号
        AccountDto accountDto = accountDAO.selectByEmail(email);
        accountDtoIsNull(accountDto);
        // 更新邮箱审核状态
        return accountDAO.updateEmailStatus(accountDto.getMediaId(), AccountDto.EMAIL_STATUS_CHECKED)>0;
    }
}
