package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import cn.com.duiba.tuia.media.api.constant.ActivityConstant;
import cn.com.duiba.tuia.media.api.constant.ActivityStatusConstant;
import cn.com.duiba.tuia.media.api.constant.MaterialSpecificationConstant;
import cn.com.duiba.tuia.media.api.constant.SplitConstant;
import cn.com.duiba.tuia.media.api.dto.MediaAppDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqActivityDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqIdAndType;
import cn.com.duiba.tuia.media.api.dto.req.ReqMaterialSpecificationDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspActivityDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationAssortDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationItemContentDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationItemDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMsContentGroupDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.CollectionUtil;
import cn.com.duiba.tuia.media.dao.ActivityDAO;
import cn.com.duiba.tuia.media.dao.MaterialSpecificationItemContentDAO;
import cn.com.duiba.tuia.media.dataobject.ActivityManualPlanDO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationItemContentDO;
import cn.com.duiba.tuia.media.domain.ActivityDto;
import cn.com.duiba.tuia.media.parser.ActivityDtoParser;
import cn.com.duiba.tuia.media.service.ActivityManualService;
import cn.com.duiba.tuia.media.service.ActivityService;
import cn.com.duiba.tuia.media.service.ActivitySortService;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import cn.com.duiba.tuia.media.service.MediaAppService;
import cn.com.duiba.tuia.media.service.cacheservice.ActivityCacheService;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;

/**
 * @author xuyenan
 * @createTime 2016/12/21
 */
@Service("activityService")
public class ActivityServiceImpl implements ActivityService {

    @Autowired
    private ActivityDAO                         activityDAO;

    @Autowired
    private MaterialSpecificationItemContentDAO materialSpecificationItemContentDAO;

    @Autowired
    private ActivityCacheService                activityCacheService;

    @Autowired
    private MaterialSpecificationService        materialSpecificationService;

    @Autowired
    private ActivitySortService                 activitySortService;

    @Autowired
    private MediaAppService                     mediaAppService;

    @Autowired
    private ActivityManualService               activityManualService;

    @Value("${media.activity.engine.url.prefix}")
    private String                              activityEngineUrlPrefix;

    /** The logger. */
    private Logger                              logger = LoggerFactory.getLogger(ActivityServiceImpl.class);

    @Override
    public List<RspActivityDto> getPlanList(ReqActivityDto req) throws TuiaMediaException {
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
        List<ActivityDto> result = activityDAO.getList(req);
        List<RspActivityDto> list = new ArrayList<>();
        for (ActivityDto activityDto : result) {
            RspActivityDto rspActivityDto = ActivityDtoParser.parseToRspActivityDto(activityDto);
            // 查询素材数量
            rspActivityDto.setItemContentAmount(materialSpecificationItemContentDAO.itemContentAmountBySource(rspActivityDto.getId(),
                                                                                                              rspActivityDto.getSource()));
            list.add(rspActivityDto);
        }
        return list;
    }

    @Override
    public int getPlanListAmount(ReqActivityDto req) throws TuiaMediaException {
        return activityDAO.getListAmount(req);
    }

    @Transactional
    @Override
    public Boolean addActivityPlan(List<ReqIdAndType> activities, Integer actSource) throws TuiaMediaException {
        for (ReqIdAndType idAndType : activities) {
            ActivityDto activityDto = new ActivityDto();
            activityDto.setActivityId(idAndType.getId());
            activityDto.setActivityType(idAndType.getType());
            activityDto.setSource(actSource);
            activityDto.setDirectAdvertMode(ActivityConstant.DIRECT_ADVERT_MODE_ONLY);
            activityDto.setIsPublish(ActivityConstant.IS_PUBLISH);
            if (ReqIdAndType.REQ_ACT_SOURCE_DUIBA == actSource) {
                activityDto.setIsPublish(ActivityConstant.IS_PUBLISH); // 未删除
                activityDAO.update(activityDto);
            }

            if (ReqIdAndType.REQ_ACT_SOURCE_TUIA == actSource) {
                activityDto.setActivityName(idAndType.getName());
                activityDto.setActivityStatus(ActivityStatusConstant.STATUS_OPEN);
                activityDto.setSource(ReqIdAndType.REQ_ACT_SOURCE_TUIA);
                // 先判断数据库是否有数据
                ActivityDto dto = activityDAO.getDetailBySource(activityDto.getActivityId(), activityDto.getSource());
                if (dto == null) {
                    activityDAO.insert(activityDto);
                } else {
                    activityDto.setIsPublish(ActivityConstant.IS_PUBLISH); // 未删除
                    activityDAO.update(activityDto);
                }
            }

        }
        return true;
    }

    @Transactional
    @Override
    public Boolean updateActivityPlan(ReqActivityDto reqActivityDto) throws TuiaMediaException {
        // 更新活动关联规格
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(reqActivityDto.getId());
        activityDto.setActivityType(reqActivityDto.getType());
        activityDto.setSource(reqActivityDto.getSource());
        activityDto.setActivityMsList(Joiner.on(SplitConstant.SPLIT_COMMA).appendTo(new StringBuilder(),
                                                                                    reqActivityDto.getMsIdList()).toString());
        activityDAO.update(activityDto);
        // 更新素材列表
        List<RspMsContentGroupDto> msContentList = reqActivityDto.getMsContentList();
        // 需要删除的素材ID列表
        List<Long> needDeleteIds = new ArrayList<>();
        // 需要添加的素材列表
        List<MaterialSpecificationItemContentDO> needAddList = new ArrayList<>();
        // 查询数据库已有的规格ID列表
        /*List<Long> exitsMsIds = materialSpecificationItemContentDAO.selectMsIdsByActivity(reqActivityDto.getId(),
                                                                                          reqActivityDto.getType());*/
        List<Long> exitsMsIds = materialSpecificationItemContentDAO.selectMsIdsByActivityAndSource(reqActivityDto.getId(),
                                                                                                   reqActivityDto.getSource());
        // 前端传过来的规格列表
        List<Long> newMsIds = CollectionUtil.getFieldList(msContentList, "msId");
        
        //TODO 改造循环调用数据库
        for (Long msId : exitsMsIds) {
            if (!newMsIds.contains(msId)) {
                // 删除已经被删除的规格
               /* materialSpecificationItemContentDAO.deleteByActivityAndMsId(reqActivityDto.getId(),
                                                                            reqActivityDto.getType(), msId);*/
                
                materialSpecificationItemContentDAO.deleteByActivityAndMsIdAndSource(reqActivityDto.getId(),
                                                                                     reqActivityDto.getSource(), msId);
            }
        }
        // 遍历某一规格对应的所有创意
        for (RspMsContentGroupDto msContentGroupDto : msContentList) {
           /* List<MaterialSpecificationItemContentDO> msItemContentList = materialSpecificationItemContentDAO.selectByActivityAndMsId(reqActivityDto.getId(),
                                                                                                                                     reqActivityDto.getType(),
                                                                                                                                     msContentGroupDto.getMsId());*/
            
            List<MaterialSpecificationItemContentDO> msItemContentList = materialSpecificationItemContentDAO.selectByActivityAndMsIdAndSource(reqActivityDto.getId(),
                                                                                                                                              reqActivityDto.getSource(),
                                                                                                                                              msContentGroupDto.getMsId());
            List<Long> ids = CollectionUtil.getFieldList(msItemContentList, "id");
            List<RspMaterialSpecificationAssortDto> itemContentList = msContentGroupDto.getItemContentGroupList();
            for (RspMaterialSpecificationAssortDto itemContentGroupDto : itemContentList) {
                // 如果没有修改,则不删除
                if (itemContentGroupDto.getId() != null) {
                    ids.remove(itemContentGroupDto.getId());
                } else {
                    // 构造创意
                    MaterialSpecificationItemContentDO materialSpecificationItemContentDO = new MaterialSpecificationItemContentDO();
                    materialSpecificationItemContentDO.setMsId(msContentGroupDto.getMsId());
                    materialSpecificationItemContentDO.setActivityId(reqActivityDto.getId());
                    materialSpecificationItemContentDO.setActivityType(reqActivityDto.getType());
                    materialSpecificationItemContentDO.setSource(reqActivityDto.getSource());
                    List<String> contentList = new ArrayList<>();
                    for (RspMaterialSpecificationItemContentDto msItemContentDto : itemContentGroupDto.getValue()) {
                        if (MaterialSpecificationConstant.ITEM_TYPE_IMAGE == msItemContentDto.getItemType()) {
                            contentList.add(msItemContentDto.getMsItemId() + SplitConstant.SPLIT_COMMA
                                            + msItemContentDto.getImageUrl());
                        } else {
                            contentList.add(msItemContentDto.getMsItemId() + SplitConstant.SPLIT_COMMA
                                            + msItemContentDto.getLetter());
                        }
                    }
                    StringBuilder sb = new StringBuilder();
                    Joiner.on(SplitConstant.SPLIT_SEMICOLON).appendTo(sb, contentList);
                    materialSpecificationItemContentDO.setContent(sb.toString());
                    needAddList.add(materialSpecificationItemContentDO);
                }
            }
            needDeleteIds.addAll(ids);
        }
        materialSpecificationItemContentDAO.deleteBatch(needDeleteIds);
        materialSpecificationItemContentDAO.insertBatch(needAddList);
        return true;
    }

    @Override
    public RspActivityDto getActivityPlanDetail(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = activityCacheService.get(activityId, activityType);
        if (activityDto == null || activityDto.getSource() == null) {
            activityDto = activityDAO.getDetail(activityId, activityType);
            if (activityDto != null) {
                activityCacheService.set(activityDto);
            }
        }
        if (activityDto == null) {
            logger.info(" getActivityPlanDetail get activityDto is null, the activityId = [{}] the activityType = [{}]",
                        activityId, activityType);
            throw new TuiaMediaException(ErrorCode.E0501004);
        }
        return ActivityDtoParser.parseToRspActivityDto(activityDto);
    }
    
    @Override
    public RspActivityDto getActivityPlanDetailBySource(Long activityId, Integer source) throws TuiaMediaException{
        ActivityDto activityDto = activityCacheService.getBySource(activityId, source);
        if (activityDto == null || activityDto.getSource() == null) {
            activityDto = activityDAO.getDetailBySource(activityId, source);
            if (activityDto != null) {
                activityCacheService.set(activityDto);
            }
        }
        if (activityDto == null) {
            logger.info(" getActivityPlanDetailBySource get activityDto is null, the activityId = [{}] the source = [{}]",
                        activityId, source);
            throw new TuiaMediaException(ErrorCode.E0501004);
        }
        return ActivityDtoParser.parseToRspActivityDto(activityDto);
    }

    @Transactional
    @Override
    public Boolean changeActivityPlanEnable(Long activityId, Integer activityType, Integer enableStatus)
                                                                                                        throws TuiaMediaException {
        if (ActivityConstant.IS_NOT_ENABLE != enableStatus && ActivityConstant.IS_ENABLE != enableStatus) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        // 人工排序列表同步活动
        if (ActivityConstant.IS_ENABLE == enableStatus) {
            ReqIdAndType reqIdAndType = new ReqIdAndType();
            reqIdAndType.setId(activityId);
            reqIdAndType.setType(activityType);
            activitySortService.addActivitySort(0L, Lists.newArrayList(reqIdAndType));
        } else {
            activitySortService.deleteActivity(activityId, activityType, 0L);
        }
        // 更新活动计划状态
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsEnable(enableStatus);
        return activityDAO.update(activityDto) == 1;
    }
    
    @Transactional
    @Override
    public Boolean changeActivityPlanEnableBySource(Long activityId, Integer source, Integer enableStatus) throws TuiaMediaException{
        if (ActivityConstant.IS_NOT_ENABLE != enableStatus && ActivityConstant.IS_ENABLE != enableStatus) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        // 人工排序列表同步活动
        if (ActivityConstant.IS_ENABLE == enableStatus) {
            ReqIdAndType reqIdAndType = new ReqIdAndType();
            reqIdAndType.setId(activityId);
            reqIdAndType.setSource(source);
            //TODO 修改activitySortService.addActivitySort的逻辑
            activitySortService.addActivitySort(0L, Lists.newArrayList(reqIdAndType));
        } else {
            activitySortService.deleteActivity(activityId, source, 0L);
        }
        // 更新活动计划状态
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setSource(source);
        activityDto.setIsEnable(enableStatus);
        return activityDAO.update(activityDto) == 1;
        
    }

    @Transactional
    @Override
    public Boolean deleteActivityPlan(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsPublish(ActivityConstant.IS_NOT_PUBLISH);
        // 删除所有定制列表中的该活动计划
        activitySortService.deleteActivity(activityId, activityType, null);
        return activityDAO.update(activityDto) == 1;
    }
    
    @Override
    public Boolean deleteActivityPlanBySource(Long activityId, Integer source) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(source);
        activityDto.setIsPublish(ActivityConstant.IS_NOT_PUBLISH);
        // 删除所有定制列表中的该活动计划
        activitySortService.deleteActivityBySource(activityId, source, null);
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public List<RspActivityDto> getActivityByIds(List<Long> ids, Integer actSource) throws TuiaMediaException {
        List<ActivityDto> result = activityDAO.getByIds(ids, actSource);
        List<RspActivityDto> list = new ArrayList<>();
        for (ActivityDto activityDto : result) {
            list.add(ActivityDtoParser.parseToRspActivityDto(activityDto));
        }
        return list;
    }

    @Override
    public Boolean insertActivity(ActivityDto activityDto) throws TuiaMediaException {
        return activityDAO.insert(activityDto) == 1;
    }

    @Override
    public Boolean updateActivity(ActivityDto activityDto) throws TuiaMediaException {
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public Boolean deleteActivity(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsDelete(ActivityConstant.IS_DELETE);
        return activityDAO.update(activityDto) == 1;
    }
    
    @Override
    public Boolean deleteActivityBySource(Long activityId, Integer source) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setSource(source);
        activityDto.setIsDelete(ActivityConstant.IS_DELETE);
        return activityDAO.update(activityDto) == 1;
    }

    @Override
    public String getActivityName(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = activityDAO.getDetail(activityId, activityType);
        if (activityDto != null) {
            return activityDto.getActivityName();
        }
        return null;
    }
    
    @Override
    public String getActivityNameBySource(Long activityId, Integer source) throws TuiaMediaException {
        ActivityDto activityDto = activityDAO.getDetailBySource(activityId, source);
        if (activityDto != null) {
            return activityDto.getActivityName();
        }
        return null;
    }

    //TODO 调用的地方需改造
    @Override
    public RspActivityDto getMsItemContent(RspActivityDto rspActivityDto) throws TuiaMediaException {
        if (rspActivityDto == null) {
            return null;
        }
        // 查询素材列表
        List<Long> msIds = rspActivityDto.getMsIdList();
        if (CollectionUtils.isEmpty(msIds)) {
            return rspActivityDto;
        }
        List<RspMaterialSpecificationDto> msItemContentList = new ArrayList<>();
        Iterator<Long> iterator = msIds.iterator();
        while (iterator.hasNext()) {
            Long msId = iterator.next();
            RspMaterialSpecificationDto rspMaterialSpecificationDto = materialSpecificationService.getMaterialSpecificationDetail(msId);
            if (MaterialSpecificationConstant.MS_DELETE == rspMaterialSpecificationDto.getIsDelete()) {
                iterator.remove();
                continue;
            }
            // 规格模板Map，下方要用到
            Map<Long, RspMaterialSpecificationItemDto> msItemMap = new HashMap<>();
            for (RspMaterialSpecificationItemDto msItem : rspMaterialSpecificationDto.getItemDtoList()) {
                msItemMap.put(msItem.getId(), msItem);
            }
           /* List<MaterialSpecificationItemContentDO> list = materialSpecificationItemContentDAO.selectByActivityAndMsId(rspActivityDto.getId(),
                                                                                                                        rspActivityDto.getType(),
                                                                                                                        msId);*/
            
            List<MaterialSpecificationItemContentDO> list = materialSpecificationItemContentDAO.selectByActivityAndMsIdAndSource(rspActivityDto.getId(),
                                                                                                                                 rspActivityDto.getSource(),
                                                                                                                                 msId);
            if (!CollectionUtils.isEmpty(list)) {
                List<RspMaterialSpecificationAssortDto> rspList = new ArrayList<>();
                for (MaterialSpecificationItemContentDO itemContentDO : list) {
                    RspMaterialSpecificationAssortDto itemContentGroup = new RspMaterialSpecificationAssortDto();
                    itemContentGroup.setId(itemContentDO.getId());
                    itemContentGroup.setIsNewContent(itemContentDO.getIsNewContent());
                    itemContentGroup.setValue(analyzeContent(itemContentDO, msItemMap));
                    rspList.add(itemContentGroup);
                }
                rspMaterialSpecificationDto.setItemContentList(rspList);
            }
            msItemContentList.add(rspMaterialSpecificationDto);
        }
        rspActivityDto.setMsList(msItemContentList);
        return rspActivityDto;
    }

    /**
     * 素材内容解析
     */
    private List<RspMaterialSpecificationItemContentDto> analyzeContent(MaterialSpecificationItemContentDO itemContentDO,
                                                                        Map<Long, RspMaterialSpecificationItemDto> msItemMap) {
        List<RspMaterialSpecificationItemContentDto> contentDtoList = new ArrayList<>();
        for (String content : itemContentDO.getContent().split(SplitConstant.SPLIT_SEMICOLON)) {
            Long msItemId = Long.parseLong(content.split(SplitConstant.SPLIT_COMMA)[0]);
            RspMaterialSpecificationItemDto msItem = msItemMap.get(msItemId);
            if (msItem != null) {
                RspMaterialSpecificationItemContentDto msItemContentDto = new RspMaterialSpecificationItemContentDto();
                String string = content.split(SplitConstant.SPLIT_COMMA)[1];
                msItemContentDto.setItemType(msItem.getItemType());
                msItemContentDto.setImageWidth(msItem.getImageWidth());
                msItemContentDto.setImageHeight(msItem.getImageHeight());
                msItemContentDto.setDescription(msItem.getDescription());
                msItemContentDto.setMsItemId(msItemId);
                msItemContentDto.setMsId(msItem.getMsId());
                if (MaterialSpecificationConstant.ITEM_TYPE_IMAGE == msItem.getItemType()) {
                    msItemContentDto.setImageUrl(string);
                    contentDtoList.add(0, msItemContentDto);
                }
                if (MaterialSpecificationConstant.ITEM_TYPE_LETTER == msItem.getItemType()) {
                    msItemContentDto.setLetter(string);
                    contentDtoList.add(msItemContentDto);
                }
            }
        }
        return contentDtoList;
    }

    @Override
    public List<Long> getIdsByName(String name) throws TuiaMediaException {
        return activityDAO.getIdsByName(name);
    }

    @Override
    public List<RspMaterialSpecificationDto> getMsItemList() throws TuiaMediaException {
        ReqMaterialSpecificationDto req = new ReqMaterialSpecificationDto();
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setPageSize(-1);
        List<RspMaterialSpecificationDto> list = materialSpecificationService.getMaterialSpecificationList(req);
        List<RspMaterialSpecificationDto> result = new ArrayList<>();
        for (RspMaterialSpecificationDto ms : list) {
            result.add(materialSpecificationService.getMaterialSpecificationDetail(ms.getId()));
        }
        return result;
    }

    @Override
    public String getActivityUrl(Long activityAppId, Long adslotId) throws TuiaMediaException {
        ActivityManualPlanDO activityManual = activityManualService.findActivityManualPlan(adslotId);
        if (activityManual == null) {
            return null;
        }

        StringBuilder activityUrl = new StringBuilder();
        MediaAppDto app = mediaAppService.getMediaAppDto(activityAppId);
        if (app != null) {
            String appKey = app.getAppKey();
            activityUrl.append(activityEngineUrlPrefix).append("?appKey=").append(appKey).append("&adslotId=").append(adslotId);
        }
        return activityUrl.toString();
    }

    @Override
    public Boolean changeToOldAct(Long activityId, Integer activityType) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setActivityType(activityType);
        activityDto.setIsNewAct(ActivityConstant.IS_NOT_NEW_ACT);
        return updateActivity(activityDto);
    }
    
    @Override
    public Boolean changeToOldActBySource(Long activityId, Integer source) throws TuiaMediaException {
        ActivityDto activityDto = new ActivityDto();
        activityDto.setActivityId(activityId);
        activityDto.setSource(source);
        activityDto.setIsNewAct(ActivityConstant.IS_NOT_NEW_ACT);
        return updateActivity(activityDto);
    }

    @Override
    public Boolean addActivityToManualList() throws TuiaMediaException {
        List<ActivityDto> list = activityDAO.selectEngineActivityList();
        list.addAll(activityDAO.selectNewActivityList());
        List<ReqIdAndType> activityList = new ArrayList<>();
        for (ActivityDto activityDto : list) {
            ReqIdAndType reqIdAndType = new ReqIdAndType();
            reqIdAndType.setId(activityDto.getActivityId());
            //reqIdAndType.setType(activityDto.getActivityType());
            reqIdAndType.setSource(activityDto.getSource());
            activityList.add(reqIdAndType);
        }
        activitySortService.addActivitySort(0L, activityList);
        return true;
    }

   
    @Override
    public Map<Long, String> getCacheActivityName(List<ReqIdAndType> activities) throws TuiaMediaException {
        Map<Long,String> activityMap = new HashMap<>();
        for (ReqIdAndType req: activities) {
            //ActivityDto activityDto=activityCacheService.get(req.getId(),req.getType());
            ActivityDto activityDto=activityCacheService.getBySource(req.getId(), req.getSource());
            if (activityDto == null || activityDto.getSource() == null) {
                activityDto = activityDAO.getDetailBySource(req.getId(), req.getSource());
                if (activityDto != null) {
                    activityCacheService.setBySource(activityDto);
                }
            }
            if (activityDto!=null) {
                activityMap.put(activityDto.getActivityId(),activityDto.getActivityName());
            }
        }
        
        return activityMap;
    }

    @Override
    public List<Long> getActivityIdsByName(String activityName) throws TuiaMediaException {
        return activityDAO.selectActivityIdsByName(activityName);
    }
    
    @Override
    public List<Long> getActivityBySource(int source, int limit) throws TuiaMediaException {
        return activityDAO.selectBySource(source, limit);
    }

    @Override
    public List<RspActivityDto> getActivityPlanDetail4List(List<Long> activityIds, List<Integer> activityTypes)
                                                                                                               throws TuiaMediaException {
        List<RspActivityDto> rspActivityDtos = Lists.newArrayList();
        if(CollectionUtils.isEmpty(activityIds)||CollectionUtils.isEmpty(activityTypes)){
            return rspActivityDtos;
        }else{
            List<ActivityDto> activityDtos = activityDAO.getDetail4List(activityIds, activityTypes);
            
            if (!CollectionUtils.isEmpty(activityDtos)) {
                for (ActivityDto activityDto : activityDtos) {
                    rspActivityDtos.add(ActivityDtoParser.parseToRspActivityDto(activityDto));
                }
            }
            return rspActivityDtos;
        }
    }

    @Override
    public List<RspActivityDto> getActivityPlanDetail4ListBySource(List<Long> activityIds, List<Integer> sources)
                                                                                                                 throws TuiaMediaException {
        List<RspActivityDto> rspActivityDtos = Lists.newArrayList();
        if(CollectionUtils.isEmpty(activityIds)||CollectionUtils.isEmpty(sources)){
            return rspActivityDtos;
        }else{
            List<ActivityDto> activityDtos = activityDAO.getDetail4ListBySource(activityIds, sources);
            
            if (!CollectionUtils.isEmpty(activityDtos)) {
                for (ActivityDto activityDto : activityDtos) {
                    rspActivityDtos.add(ActivityDtoParser.parseToRspActivityDto(activityDto));
                }
            }
            return rspActivityDtos;
        }
    }
}
