package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.tuia.media.api.constant.ActivityConstant;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqActivitySlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqCopySlot;
import cn.com.duiba.tuia.media.api.dto.req.ReqIdAndType;
import cn.com.duiba.tuia.media.api.dto.rsp.RspActivitySlotDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.CollectionUtil;
import cn.com.duiba.tuia.media.dao.ActivitySlotDAO;
import cn.com.duiba.tuia.media.dao.ActivitySortDAO;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.dao.SlotDAO;
import cn.com.duiba.tuia.media.dataobject.ActivitySlotDO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationDO;
import cn.com.duiba.tuia.media.domain.ActivitySortDto;
import cn.com.duiba.tuia.media.service.ActivitySlotService;
import cn.com.duiba.tuia.media.service.ActivitySortService;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;

import com.google.common.collect.Lists;

/**
 * @author xuyenan
 * @createTime 2016/12/21
 */
@Service("activitySlotService")
public class ActivitySlotServiceImpl implements ActivitySlotService {

    @Autowired
    private ActivitySlotDAO              activitySlotDAO;

    @Autowired
    private SlotDAO                      slotDAO;

    @Autowired
    private ActivitySortDAO              activitySortDAO;

    @Autowired
    private MediaAppDao                  mediaAppDao;

    @Autowired
    private MediaCacheService            mediaCacheService;

    @Autowired
    private MaterialSpecificationService materialSpecificationService;

    @Autowired
    private ActivitySortService          activitySortService;

    @Override
    public Boolean addSlots(List<Long> slotIds) throws TuiaMediaException {
        if (CollectionUtils.isEmpty(slotIds)) {
            return true;
        }
        List<ActivitySlotDO> list = new ArrayList<>(slotIds.size());
        for (Long slotId : slotIds) {
            ActivitySlotDO activitySlotDO = new ActivitySlotDO();
            activitySlotDO.setSlotId(slotId);
            // 过滤已经添加过的广告位
            if (activitySlotDAO.getBySlotId(slotId) == null) {
                list.add(activitySlotDO);
            }
        }
        activitySlotDAO.insertBatch(list);
        return true;
    }

    @Transactional
    @Override
    public Boolean deleteSlot(Long slotId) throws TuiaMediaException {
        activitySlotDAO.delete(slotId);
        activitySortDAO.deleteAllSlotActivity(slotId);
        return true;
    }

    @Override
    public List<RspActivitySlotDto> getList(ReqActivitySlotDto req) throws TuiaMediaException {
        // 根据名字模糊匹配广告位ID列表
        if (StringUtils.isNotEmpty(req.getName())) {
            List<Long> ids = slotDAO.selectIdsByName(req.getName());
            if (CollectionUtils.isEmpty(ids)) {
                return Lists.newArrayList();
            }
            req.setIdList(ids);
        }
        if (StringUtils.isNotBlank(req.getAppName())) {
            List<Long> appIds = mediaAppDao.selectIdsByName(req.getAppName());
            if (CollectionUtils.isEmpty(appIds)) {
                return Lists.newArrayList();
            }
            req.setAppIdList(appIds);
        }
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));

        List<RspActivitySlotDto> rspActivitySlotList = activitySlotDAO.getList(req);
        List<SlotDto> slotList = slotDAO.selectListDetail(CollectionUtil.getFieldList(rspActivitySlotList, "slotId"));
        Map<Long, MaterialSpecificationDO> msMap = materialSpecificationService.getMapByIds(CollectionUtil.getFieldList(slotList,
                                                                                                                        "slotMsId"));
        for (RspActivitySlotDto activitySlotDto : rspActivitySlotList) {
            activitySlotDto.setAppName(mediaCacheService.getAppNameByIdKey(activitySlotDto.getAppId()));
            activitySlotDto.setMsName(SlotDto.getSlotTypeNameBySlotType(activitySlotDto.getSlotType()));
            // 规格若被删除，则状态置为无效
            if (activitySlotDto.getSlotMsId() != null) {
                MaterialSpecificationDO ms = msMap.get(activitySlotDto.getSlotMsId());
                if (ms != null) {
                    activitySlotDto.setMsName(ms.getTitle());
                }
            }
        }
        return rspActivitySlotList;
    }

    @Override
    public int getListAmount(ReqActivitySlotDto req) throws TuiaMediaException {
        if (StringUtils.isNotEmpty(req.getName())) {
            List<Long> ids = slotDAO.selectIdsByName(req.getName());
            if (CollectionUtils.isEmpty(ids)) {
                return 0;
            }
            req.setIdList(ids);
        }
        if (StringUtils.isNotBlank(req.getAppName())) {
            List<Long> appIds = mediaAppDao.selectIdsByName(req.getAppName());
            if (CollectionUtils.isEmpty(appIds)) {
                return 0;
            }
            req.setAppIdList(appIds);
        }
        return activitySlotDAO.getListAmount(req);
    }

    @Override
    public ActivitySlotDO getBySlotId(Long slotId) throws TuiaMediaException {
        return activitySlotDAO.getBySlotId(slotId);
    }

    @Override
    public Boolean updateDirectMode(Long slotId, Integer directMode) throws TuiaMediaException {
        if (ActivityConstant.DIRECT_ADVERT_MODE_ONLY != directMode
            && ActivityConstant.DIRECT_ADVERT_MODE_PRIOR != directMode) {
            return false;
        }
        return activitySlotDAO.update(slotId, directMode) == 1;
    }

    //TODO
    @Override
    public Boolean copySlot(ReqCopySlot req) throws TuiaMediaException {// NOSONAR

        if (req.getParentSlotId() == req.getChildSlotId()) {
            throw new TuiaMediaException(ErrorCode.E0501005);
        }
        SlotDto slotDto = slotDAO.selectById(req.getChildSlotId());

        if (slotDto == null) {
            throw new TuiaMediaException(ErrorCode.E0304008);
        }
        // 通过广告位ID查询活动广告位
        ActivitySlotDO childActivitySlot = activitySlotDAO.getBySlotId(req.getChildSlotId());

        // 通过广告位ID查询copy的活动广告位参数
        ActivitySlotDO parentActivitySlot = activitySlotDAO.getBySlotId(req.getParentSlotId());

        List<ActivitySortDto> parentActivitySortParam = activitySortDAO.getAllActivityBySlotId(req.getParentSlotId());
        if (CollectionUtils.isEmpty(parentActivitySortParam)) {
            throw new TuiaMediaException(ErrorCode.E0501006);
        }
        List<ReqIdAndType> idAndTypeList = new ArrayList<>();
        for (ActivitySortDto dto : parentActivitySortParam) {
            ReqIdAndType reqIdAndType = new ReqIdAndType();
            reqIdAndType.setId(dto.getActivityId());
            reqIdAndType.setSource(dto.getSource());
            //reqIdAndType.setType(dto.getActivityType());
            idAndTypeList.add(reqIdAndType);
        }
        // 如果广告位存在广告位定制活动表中
        if (childActivitySlot != null) {
            List<ActivitySortDto> childActivitySortParam = activitySortDAO.getAllActivityBySlotId(req.getChildSlotId());
            if (CollectionUtils.isNotEmpty(childActivitySortParam)) {
                addActivitySlot(req, parentActivitySlot, childActivitySlot); // 修改活动广告投放类型
                activitySortService.addActivitySort(req.getChildSlotId(), idAndTypeList);
            } else if (CollectionUtils.isEmpty(childActivitySortParam)) {
                addActivitySlot(req, parentActivitySlot, childActivitySlot); // 添加活动广告
                activitySortService.addActivitySort(req.getChildSlotId(), idAndTypeList);// 复制广告位下面的活动
            }
        } else {// 如果广告位不存在广告位定制活动表中
            if (addActivitySlot(req, parentActivitySlot, childActivitySlot)) { // 先创建广告位
                activitySortService.addActivitySort(req.getChildSlotId(), idAndTypeList);// 复制广告位下面的活动
            }
        }
        return true;
    }

    /**
     * addActivitySlot:(添加活动广告信息). <br/>
     *
     * @param req
     * @param param
     * @param activitySortList
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private boolean addActivitySlot(ReqCopySlot req, ActivitySlotDO parentActivitySlotParam,
                                    ActivitySlotDO childActivitySlotParam) throws TuiaMediaException {
        if (childActivitySlotParam == null) {
            ActivitySlotDO activitySlot = new ActivitySlotDO();
            activitySlot.setSlotId(req.getChildSlotId());
            activitySlot.setDirectMode(parentActivitySlotParam.getDirectMode());
            return activitySlotDAO.insertActivitySlot(activitySlot) > 0;// 插入一个新的广告位到广告位定制列表
        } else {
            return activitySlotDAO.update(childActivitySlotParam.getSlotId(), parentActivitySlotParam.getDirectMode()) > 0;// 插入一个新的广告位到广告位定制列表
        }

    }
}
