package cn.com.duiba.tuia.media.service.impl;

import java.util.*;

import cn.com.duiba.tuia.media.api.constant.ActivityStatusConstant;
import cn.com.duiba.tuia.media.api.constant.CacheKey;
import cn.com.duiba.tuia.media.api.constant.SplitConstant;
import cn.com.duiba.tuia.media.api.dto.SlotCacheDto;
import cn.com.duiba.tuia.media.service.cacheservice.ActivityDirectAppCacheService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import cn.com.duiba.tuia.media.api.constant.ActivityConstant;
import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqActivitySort;
import cn.com.duiba.tuia.media.api.dto.req.ReqIdAndType;
import cn.com.duiba.tuia.media.api.dto.rsp.RspActivityDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.ActivityDAO;
import cn.com.duiba.tuia.media.dao.ActivitySortDAO;
import cn.com.duiba.tuia.media.dataobject.ActivitySlotDO;
import cn.com.duiba.tuia.media.dataobject.MediaAppActivityDO;
import cn.com.duiba.tuia.media.domain.ActivityDto;
import cn.com.duiba.tuia.media.domain.ActivitySortDto;
import cn.com.duiba.tuia.media.parser.ActivityDtoParser;
import cn.com.duiba.tuia.media.service.*;

/**
 * Function: ActivitySortServiceImpl
 *
 * @author xuyenan
 * @createTime 2016/10/24
 */
@Service
public class ActivitySortServiceImpl implements ActivitySortService {

    protected Logger                      logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private ActivitySortDAO               activitySortDAO;
    @Autowired
    private SlotService                   slotService;
    @Autowired
    private ActivityService               activityService;
    @Autowired
    private ActivitySlotService           activitySlotService;
    @Autowired
    private ActivityDAO                   activityDAO;
    @Autowired
    private SlotCacheService              slotCacheService;
    @Autowired
    private ActivityDirectAppCacheService activityDirectAppCacheService;

    @Override
    public List<ActivitySortDto> selectByCondition(ReqActivitySort reqActivitySort) throws TuiaMediaException {
        // 将名称模糊查询转换为ID列表
        if (!StringUtils.isEmpty(reqActivitySort.getName())) {
            List<Long> list = activityService.getIdsByName(reqActivitySort.getName());
            if (CollectionUtils.isEmpty(list)) {
                return new ArrayList<>();
            }
            reqActivitySort.setActivityIds(list);
        }
        return activitySortDAO.selectByCondition(reqActivitySort);
    }

    @Override
    public int selectByConditionAmount(ReqActivitySort reqActivitySort) throws TuiaMediaException {
        if (!StringUtils.isEmpty(reqActivitySort.getName())) {
            List<Long> list = activityService.getIdsByName(reqActivitySort.getName());
            if (CollectionUtils.isEmpty(list)) {
                return 0;
            }
            reqActivitySort.setActivityIds(list);
        }
        return activitySortDAO.selectByConditionAmount(reqActivitySort);
    }

    @Override
    @Transactional
    public boolean deleteActivity(Long activityId, Integer activityType, Long slotId) throws TuiaMediaException {
        if (activityId == null || activityType == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        List<ActivitySortDto> activitySortList = selectByCondition(reqActivitySort);
        int location = 0;
        for (ActivitySortDto activitySortDto : activitySortList) {
            if (activitySortDto.getActivityId().equals(activityId)
                && activitySortDto.getActivityType().equals(activityType)) {
                location = activitySortDto.getAbsoluteSort();
            }
        }
        activitySortDAO.deleteActivity(slotId, activityId, activityType);
        // 删除某个活动后，后面所有活动的排序值-1
        activitySortDAO.updateSortBatch(slotId, location,
                                        activitySortList.get(activitySortList.size() - 1).getAbsoluteSort(), -1);
        return true;
    }
    
    @Override
    @Transactional
    public boolean deleteActivityBySource(Long activityId, Integer source, Long slotId) throws TuiaMediaException {
        if (activityId == null || source == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        List<ActivitySortDto> activitySortList = selectByCondition(reqActivitySort);
        int location = 0;
        for (ActivitySortDto activitySortDto : activitySortList) {
            if (activitySortDto.getActivityId().equals(activityId)
                && activitySortDto.getSource().equals(source)) {
                location = activitySortDto.getAbsoluteSort();
            }
        }
        activitySortDAO.deleteActivityBySource(slotId, activityId, source);
        // 删除某个活动后，后面所有活动的排序值-1
        activitySortDAO.updateSortBatch(slotId, location,
                                        activitySortList.get(activitySortList.size() - 1).getAbsoluteSort(), -1);
        return true;
    }

    //TODO
    @Override
    public List<RspActivityDto> getActivityForRecommendByApp(Long appId) throws TuiaMediaException {
        if (appId == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        List<ActivitySortDto> sortList = activitySortDAO.getActivityForRecommendByApp(appId);
        List<RspActivityDto> result = new ArrayList<>();
        for (ActivitySortDto activitySortDto : sortList) {
            /*RspActivityDto rspActivityDto = activityService.getActivityPlanDetail(activitySortDto.getActivityId(),
                                                                                  activitySortDto.getActivityType());*/
            RspActivityDto rspActivityDto = activityService.getActivityPlanDetailBySource(activitySortDto.getActivityId(),
                                                                                          activitySortDto.getSource());
            result.add(rspActivityDto);
        }
        return result;
    }

    private List<ActivitySortDto> getActivityBySlot(Long slotId) throws TuiaMediaException {// NOSONAR
        ActivitySlotDO activitySlot = activitySlotService.getBySlotId(slotId);
        if (activitySlot == null) {
            return Collections.emptyList();
        }
        return activitySortDAO.getActivityForRecommendBySlot(slotId);
    }

    //TODO
    private boolean availableMediaApp(Long appId, RspActivityDto activity) throws TuiaMediaException {
        if (activity == null || appId == null) {
            return false;
        }

        if (ActivityConstant.IS_DIRECT_MEDIA == activity.getIsDirectMedia()) {
            //List<Long> directMediaApp = activityDirectAppCacheService.get(activity.getId(), activity.getType());
            List<Long> directMediaApp = activityDirectAppCacheService.getBySource(activity.getId(), activity.getSource());
            if (CollectionUtils.isEmpty(directMediaApp)) {
                return true;
            }
            return directMediaApp.contains(appId);
        }
        return true;
    }

    private boolean availableMediaApp(Long appId, RspActivityDto activity, Map<String, List<Long>> directMap)
                                                                                                             throws TuiaMediaException {
        if (activity == null || appId == null) {
            return false;
        }

        if (ActivityConstant.IS_DIRECT_MEDIA == activity.getIsDirectMedia()) {
            //String key = activity.getId() + SplitConstant.SPLIT_HYPHEN + activity.getType();
            String key = activity.getId() + SplitConstant.SPLIT_HYPHEN + activity.getSource();
            List<Long> directMediaApp = directMap.get(key);
            if (CollectionUtils.isEmpty(directMediaApp)) {
                return true;
            }
            return directMediaApp.contains(appId);
        }
        return true;
    }

    @Override
    public List<RspActivityDto> getActivityForRecommendBySlot(Long slotId) throws TuiaMediaException {// NOSONAR
        if (slotId == null) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }

        try {
            SlotDto slot = slotService.selectById(slotId);
            if (slot == null) {
                logger.error("Not found slot, slotId=" + slotId);
                throw new TuiaMediaException(ErrorCode.E0001005);
            }

            List<ActivitySortDto> sortList = getActivityBySlot(slotId);
            List<RspActivityDto> result = new ArrayList<>();
            for (ActivitySortDto activitySortDto : sortList) {
                //TODO
                RspActivityDto rspActivityDto = activityService.getActivityPlanDetailBySource(activitySortDto.getActivityId(),
                                                                                              activitySortDto.getSource());
                if (rspActivityDto != null && rspActivityDto.getActivityStatus() == ActivityStatusConstant.STATUS_OPEN
                    && rspActivityDto.getIsEnable() == ActivityConstant.IS_ENABLE
                    && rspActivityDto.getIsPublish() == ActivityConstant.IS_PUBLISH) {
                    if (slot.getSlotType() == SlotDto.ADSENSE_TYPE_MATERIAL_SPEC
                        && rspActivityDto.getMsIdList() != null
                        && rspActivityDto.getMsIdList().contains(slot.getSlotMsId())) {
                        if (availableMediaApp(slot.getAppId(), rspActivityDto)) {
                            // 广告位规格和活动规格匹配
                            result.add(rspActivityDto);
                        }
                    } else if (slot.getSlotType() == SlotDto.ADSENSE_TYPE_MANUAL) {
                        // 手动投放类型，不走SDK接口
                    } else if (slot.getSlotType() != SlotDto.ADSENSE_TYPE_MATERIAL_SPEC) {
                        if (availableMediaApp(slot.getAppId(), rspActivityDto)) {
                            result.add(rspActivityDto);
                        }
                    }
                }
            }
            return result;
        } catch (Exception e) {
            logger.error("ActivitySortService.getActivityForRecommendBySlot is error");
            throw new TuiaMediaException(ErrorCode.E9999999, e);
        }
    }
    
    //TODO 添加source修改
    @Override
    public int addActivitySort(Long slotId, List<ReqIdAndType> idAndTypes) {// NOSONAR
        try {
            // 查询已有活动列表
            ReqActivitySort reqActivitySort = new ReqActivitySort();
            reqActivitySort.setSlotId(slotId);
            List<ActivitySortDto> existList = selectByCondition(reqActivitySort);
            Set<String> existActivity = new HashSet<>(existList.size());
            for (ActivitySortDto activitySortDto : existList) {
               /* existActivity.add(activitySortDto.getActivityId() + SplitConstant.SPLIT_HYPHEN
                                  + activitySortDto.getActivityType());*/
                //TODO
                existActivity.add(CacheKey.ACTIVITY_SOURCE_KEY+activitySortDto.getActivityId() + SplitConstant.SPLIT_HYPHEN
                                  + activitySortDto.getSource());
            }
            // 计算下一个排序值
            int sort = existList.size() == 0 ? 1 : existList.get(existList.size() - 1).getAbsoluteSort() + 1;

            List<ActivitySortDto> activitySortDtoList = new ArrayList<>();
            // 查询广告位所属媒体
            SlotDto slotDto = null;
            if (!slotId.equals(0L)) {
                slotDto = slotService.selectById(slotId);
            }
            for (ReqIdAndType idAndType : idAndTypes) {
                // 定向媒体的活动只能添加到定向的媒体中
                if (slotDto != null) {
                    // 检查活动是否开启媒体定向
                   /* RspActivityDto rspActivityDto = activityService.getActivityPlanDetail(idAndType.getId(),
                                                                                          idAndType.getType());*/
                    RspActivityDto rspActivityDto = activityService.getActivityPlanDetailBySource(idAndType.getId(),
                                                                                                  idAndType.getSource());
                    if (!availableMediaApp(slotDto.getAppId(), rspActivityDto)) {
                        continue;
                    }
                }
                // 如果已经添加过则不添加
                ReqIdAndType reqIdAndType = new ReqIdAndType();
                reqIdAndType.setId(idAndType.getId());
                reqIdAndType.setSource(idAndType.getSource());
                if (!existActivity.contains(CacheKey.ACTIVITY_SOURCE_KEY + reqIdAndType.getId()
                                            + SplitConstant.SPLIT_HYPHEN + reqIdAndType.getSource())) {
                    ActivitySortDto activitySortDto = new ActivitySortDto();
                    activitySortDto.setActivitySlotId(slotId);
                    activitySortDto.setActivityId(idAndType.getId());
                    activitySortDto.setActivityType(idAndType.getType());
                    activitySortDto.setAbsoluteSort(sort);
                    activitySortDto.setSource(idAndType.getSource());
                    activitySortDtoList.add(activitySortDto);
                    sort++;
                }
            }
            if (!activitySortDtoList.isEmpty()) {
                return activitySortDAO.insertBatch(activitySortDtoList);
            }
            return 0;
        } catch (Exception e) {
            logger.error("ActivitySortService.addActivitySort is error", e);
            return 0;
        }
    }

    @Override
    public int addDefaultActivitySort(Long slotId) {
        try {
            // 查询默认活动列表
            List<ReqIdAndType> idAndTypes = new ArrayList<>();
            ReqActivitySort condition = new ReqActivitySort();
            condition.setSlotId(0L);
            List<ActivitySortDto> needList = selectByCondition(condition);
            for (ActivitySortDto activitySortDto : needList) {
                ReqIdAndType idAndType = new ReqIdAndType();
                idAndType.setId(activitySortDto.getActivityId());
                idAndType.setType(activitySortDto.getActivityType());
                idAndType.setSource(activitySortDto.getSource());
                idAndTypes.add(idAndType);
            }
            return addActivitySort(slotId, idAndTypes);
        } catch (Exception e) {
            logger.error("ActivitySortService.addDefaultActivitySort is error", e);
            return 0;
        }
    }

    @Transactional
    @Override
    public boolean sortActivity(Long slotId, Long activityId, Integer activityType, int destLocation)
                                                                                                     throws TuiaMediaException {
        if (slotId == null || activityId == null || activityType == null || destLocation <= 0) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        List<ActivitySortDto> activitySortList = selectByCondition(reqActivitySort);
        if (destLocation > activitySortList.size()) {
            throw new TuiaMediaException(ErrorCode.E0501002);
        }
        // 原始位置计算
        int originalLoc = 0;
        for (int i = 0; i < activitySortList.size(); i++) {
            if (activitySortList.get(i).getActivityId().equals(activityId)
                && activitySortList.get(i).getActivityType().equals(activityType)) {
                originalLoc = i + 1;
                break;
            }
        }
        // 若原始位置仍旧为0，则表示该活动同时被另一操作删除，提示刷新列表
        if (originalLoc == 0) {
            throw new TuiaMediaException(null, String.format("活动[%s]已被删除，请刷新定制列表", activityId));
        }
        if (originalLoc < destLocation) {
            activitySortDAO.updateSortBatch(slotId, originalLoc + 1, destLocation, -1);
        } else if (originalLoc > destLocation) {
            activitySortDAO.updateSortBatch(slotId, destLocation, originalLoc, 1);
        }
        activitySortDAO.updateSort(slotId, activityId, activityType, destLocation);
        return true;
    }
    
    @Override
    public boolean sortActivityBySource(Long slotId, Long activityId, Integer source, int destLocation)
                                                                                                       throws TuiaMediaException {
        if (slotId == null || activityId == null || source == null || destLocation <= 0) {
            throw new TuiaMediaException(ErrorCode.E0001005);
        }
        ReqActivitySort reqActivitySort = new ReqActivitySort();
        reqActivitySort.setSlotId(slotId);
        List<ActivitySortDto> activitySortList = selectByCondition(reqActivitySort);
        if (destLocation > activitySortList.size()) {
            throw new TuiaMediaException(ErrorCode.E0501002);
        }
        // 原始位置计算
        int originalLoc = 0;
        for (int i = 0; i < activitySortList.size(); i++) {
            if (activitySortList.get(i).getActivityId().equals(activityId)
                && activitySortList.get(i).getSource().equals(source)) {
                originalLoc = i + 1;
                break;
            }
        }
        // 若原始位置仍旧为0，则表示该活动同时被另一操作删除，提示刷新列表
        if (originalLoc == 0) {
            throw new TuiaMediaException(null, String.format("活动[%s]已被删除，请刷新定制列表", activityId));
        }
        if (originalLoc < destLocation) {
            activitySortDAO.updateSortBatch(slotId, originalLoc + 1, destLocation, -1);
        } else if (originalLoc > destLocation) {
            activitySortDAO.updateSortBatch(slotId, destLocation, originalLoc, 1);
        }
        activitySortDAO.updateSortBySource(slotId, activityId, source, destLocation);
        return true;
    }

    //TODO
    @Override
    public List<RspActivityDto> getActivityForCenter(Long slotId) throws TuiaMediaException {
        List<ActivitySortDto> activitySortList = activitySortDAO.getActivityForRecommendBySlot(0L);
        List<Long> activitySortIds = Lists.newArrayList();
        //List<Integer> activitySortTypes = Lists.newArrayList();
        List<Integer> activitySortSources = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(activitySortList)) {
            for (ActivitySortDto activitySortDto : activitySortList) {
                activitySortIds.add(activitySortDto.getActivityId());
                //activitySortTypes.add(activitySortDto.getActivityType());
                activitySortSources.add(activitySortDto.getSource());
            }
        }

        /*List<RspActivityDto> rspActivityDtos = activityService.getActivityPlanDetail4List(activitySortIds,
                                                                                          activitySortTypes);*/
        List<RspActivityDto> rspActivityDtos = activityService.getActivityPlanDetail4ListBySource(activitySortIds,
                                                                                                  activitySortSources);
        Map<String, RspActivityDto> repMap = Maps.newHashMap();
        for (RspActivityDto dto : rspActivityDtos) {
            //String key = dto.getId() + SplitConstant.SPLIT_HYPHEN + dto.getType();
            String key = dto.getId() + SplitConstant.SPLIT_HYPHEN + dto.getSource();
            repMap.put(key, dto);
        }

        SlotCacheDto slotCacheDto = slotCacheService.getSlot(slotId);

        /*List<MediaAppActivityDO> mediaAppActivityDOs = activityDirectAppCacheService.get4List(activitySortIds,
                                                                                              activitySortTypes);*/
        
        List<MediaAppActivityDO> mediaAppActivityDOs = activityDirectAppCacheService.get4ListBySources(activitySortIds,
                                                                                                       activitySortSources);
        Map<String, List<Long>> directMap = Maps.newHashMap();
        for (MediaAppActivityDO mediaAppActivityDO : mediaAppActivityDOs) {
            //String key = mediaAppActivityDO.getActivityId() + SplitConstant.SPLIT_HYPHEN + mediaAppActivityDO.getActivityType();
            String key = mediaAppActivityDO.getActivityId() + SplitConstant.SPLIT_HYPHEN + mediaAppActivityDO.getSource();
            List<Long> ids = directMap.get(key);
            if (ids == null) {
                ids = new ArrayList<>();
            }
            ids.add(mediaAppActivityDO.getMediaAppId());
            directMap.put(key, ids);
        }

        List<RspActivityDto> rspActivityList = new ArrayList<>();
        for (ActivitySortDto activitySortDto : activitySortList) {
            /*String key = activitySortDto.getActivityId() + SplitConstant.SPLIT_HYPHEN
                         + activitySortDto.getActivityType();*/
            String key = activitySortDto.getActivityId() + SplitConstant.SPLIT_HYPHEN
                    + activitySortDto.getSource();
            RspActivityDto activityDto = repMap.get(key);
            if (repMap.get(key).getSource() == ReqIdAndType.REQ_ACT_SOURCE_DUIBA
                && availableMediaApp(slotCacheDto == null ? null : slotCacheDto.getAppId(), activityDto, directMap)) {
                rspActivityList.add(activityDto);
            }
        }

        return rspActivityList;
    }

    @Override
    public List<RspActivityDto> getEngineActivityList() throws TuiaMediaException {
        List<ActivityDto> list = activityDAO.selectEngineActivityList();
        List<RspActivityDto> result = new ArrayList<>();
        if (!CollectionUtils.isEmpty(list)) {
            for (ActivityDto activity : list) {
                result.add(ActivityDtoParser.parseToRspActivityDto(activity));
            }
        }
        return result;
    }

    @Override
    public List<RspActivityDto> getNewActivityList() throws TuiaMediaException {
        List<ActivityDto> list = activityDAO.selectNewActivityList();
        List<RspActivityDto> result = new ArrayList<>();
        if (!CollectionUtils.isEmpty(list)) {
            for (ActivityDto activity : list) {
                result.add(ActivityDtoParser.parseToRspActivityDto(activity));
            }
        }
        return result;
    }

    @Override
    public List<RspActivityDto> getDefaultActivityList() throws TuiaMediaException {
        List<ActivitySortDto> activitySortList = activitySortDAO.getActivityForRecommendBySlot(0L);
        List<RspActivityDto> rspActivityList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(activitySortList)) {
            for (ActivitySortDto activitySortDto : activitySortList) {
                RspActivityDto rspActivityDto = new RspActivityDto();
                rspActivityDto.setId(activitySortDto.getActivityId());
                rspActivityDto.setType(activitySortDto.getActivityType());
                rspActivityDto.setSource(activitySortDto.getSource());
                rspActivityList.add(rspActivityDto);
            }
        }
        return rspActivityList;
    }

    @Override
    public List<ActivitySortDto> getBySlot(Long slotId) throws TuiaMediaException {
        return activitySortDAO.selectBySlot(slotId);
    }

    @Override
    public Boolean updateSort(Long slotId, Long activityId, Integer activityType, int sort) throws TuiaMediaException {
        return activitySortDAO.updateSort(slotId, activityId, activityType, sort) > 1;
    }

    @Override
    public Boolean updateSortBySource(Long slotId, Long activityId, Integer source, int sort) throws TuiaMediaException {
        return activitySortDAO.updateSortBySource(slotId, activityId, source, sort) > 1;
    }

}
