package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.com.duiba.tuia.media.dao.MaterialSpecificationItemContentDAO;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import cn.com.duiba.tuia.media.api.constant.MaterialSpecificationConstant;
import cn.com.duiba.tuia.media.api.dto.req.ReqMaterialSpecificationDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationItemDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.dao.MaterialSpecificationDAO;
import cn.com.duiba.tuia.media.dao.MaterialSpecificationItemDAO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationDO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationItemContentDO;
import cn.com.duiba.tuia.media.dataobject.MaterialSpecificationItemDO;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import cn.com.duiba.tuia.media.service.SlotService;

/**
 * @author xuyenan
 * @createTime 2016/12/20
 */
@Service("materialSpecificationService")
public class MaterialSpecificationServiceImpl implements MaterialSpecificationService {

    @Autowired
    private MaterialSpecificationDAO            materialSpecificationDAO;

    @Autowired
    private MaterialSpecificationItemDAO        materialSpecificationItemDAO;

    @Autowired
    private SlotService                         slotService;

    @Autowired
    private MaterialSpecificationItemContentDAO materialSpecificationItemContentDAO;

    @Value("${media.material.specification.id}")
    private String                              materialSpecificationId;

    @Override
    public List<RspMaterialSpecificationDto> getMaterialSpecificationList(ReqMaterialSpecificationDto req)
                                                                                                          throws TuiaMediaException {
        req.setSort("gmt_modified");
        req.setOrder("desc");
        req.setRowStart(req.getPageSize() * (req.getCurrentPage() - 1));
        List<MaterialSpecificationDO> list = materialSpecificationDAO.getList(req);
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<RspMaterialSpecificationDto> result = new ArrayList<>(list.size());

        String[] ids = materialSpecificationId.split(",");
        HashMap<Long, Boolean> map = new HashMap<>();
        for (String id : ids) {
            map.put(Long.valueOf(id), true);
        }
        for (MaterialSpecificationDO materialSpecificationDO : list) {
            RspMaterialSpecificationDto rspMaterialSpecificationDto = new RspMaterialSpecificationDto();
            BeanUtils.copyProperties(materialSpecificationDO, rspMaterialSpecificationDto);
            if (map.get(rspMaterialSpecificationDto.getId()) != null && map.get(rspMaterialSpecificationDto.getId())) {
                rspMaterialSpecificationDto.setIsfixedData(MaterialSpecificationConstant.IS_FIXED_DATA);
            } else {
                rspMaterialSpecificationDto.setIsfixedData(MaterialSpecificationConstant.NO_FIXED_DATA);
            }
            result.add(rspMaterialSpecificationDto);
        }
        return result;
    }

    @Override
    public int getMaterialSpecificationAmount(ReqMaterialSpecificationDto req) throws TuiaMediaException {
        return materialSpecificationDAO.getAmount(req);
    }

    @Transactional
    @Override
    public Boolean deleteMaterialSpecification(Long msId) throws TuiaMediaException {
        // 将所有该规格的广告位都关闭
        slotService.closeSlotByMsId(msId);
        return materialSpecificationDAO.delete(msId) == 1;
    }

    @Transactional
    @Override
    public Boolean addMaterialSpecification(ReqMaterialSpecificationDto req) throws TuiaMediaException {
        MaterialSpecificationDO materialSpecificationDO = new MaterialSpecificationDO();
        materialSpecificationDO.setTitle(req.getTitle());
        materialSpecificationDO.setDescription(req.getDescription());
        materialSpecificationDO.setSchematicImg(req.getSchematicImg());
        // 如果前端未传id，表示是新增；反之则为更新
        if (req.getId() == null) {
            materialSpecificationDO.setIsDirect(MaterialSpecificationConstant.MS_NOT_DIRECT);
            materialSpecificationDAO.insert(materialSpecificationDO);
        } else {
            materialSpecificationDO.setId(req.getId());
            materialSpecificationDAO.update(materialSpecificationDO);
        }
        // 更新素材列表
        List<RspMaterialSpecificationItemDto> msItemList = req.getItemDtoList();
        List<MaterialSpecificationItemDO> list = new ArrayList<>(msItemList.size());
        for (RspMaterialSpecificationItemDto msItem : msItemList) {
            MaterialSpecificationItemDO materialSpecificationItemDO = new MaterialSpecificationItemDO();
            BeanUtils.copyProperties(msItem, materialSpecificationItemDO);
            if (msItem.getId() != null) {
                materialSpecificationItemDAO.update(materialSpecificationItemDO);
            } else {
                materialSpecificationItemDO.setMsId(materialSpecificationDO.getId());
                list.add(materialSpecificationItemDO);
            }
        }
        if (!CollectionUtils.isEmpty(list)) {
            materialSpecificationItemDAO.insertBatch(list);
        }
        return true;
    }

    @Override
    public RspMaterialSpecificationDto getMaterialSpecificationDetail(Long msId) throws TuiaMediaException {
        MaterialSpecificationDO materialSpecificationDO = materialSpecificationDAO.getById(msId);
        if (materialSpecificationDO == null) {
            throw new TuiaMediaException(ErrorCode.E0501003);
        }
        // 封装返回对象
        RspMaterialSpecificationDto result = new RspMaterialSpecificationDto();
        BeanUtils.copyProperties(materialSpecificationDO, result);
        // 获取规格对应的素材列表
        List<MaterialSpecificationItemDO> msItemList = materialSpecificationItemDAO.getByMsId(materialSpecificationDO.getId());
        if (!CollectionUtils.isEmpty(msItemList)) {
            List<RspMaterialSpecificationItemDto> rspMsItemList = new ArrayList<>(msItemList.size());
            for (MaterialSpecificationItemDO msItem : msItemList) {
                RspMaterialSpecificationItemDto rspMsItem = new RspMaterialSpecificationItemDto();
                BeanUtils.copyProperties(msItem, rspMsItem);
                if (MaterialSpecificationConstant.ITEM_TYPE_IMAGE == rspMsItem.getItemType()) {
                    rspMsItemList.add(0, rspMsItem);
                } else {
                    rspMsItemList.add(rspMsItemList.size(), rspMsItem);
                }
            }
            result.setItemDtoList(rspMsItemList);
        }
        return result;
    }

    @Override
    public Map<Long, String> getMsNameByIds(List<Long> ids) throws TuiaMediaException {
        if (CollectionUtils.isEmpty(ids)) {
            return new HashMap<>();
        }
        List<MaterialSpecificationDO> materialSpecificationDOList = materialSpecificationDAO.getByIds(ids);
        Map<Long, String> map = new HashMap<>();
        for (MaterialSpecificationDO materialSpecificationDO : materialSpecificationDOList) {
            map.put(materialSpecificationDO.getId(), materialSpecificationDO.getTitle());
        }
        return map;
    }

    @Override
    public Map<Long, MaterialSpecificationDO> getMapByIds(List<Long> ids) throws TuiaMediaException {
        List<MaterialSpecificationDO> materialSpecificationDOList = materialSpecificationDAO.getByIds(ids);
        Map<Long, MaterialSpecificationDO> map = new HashMap<>();
        for (MaterialSpecificationDO materialSpecificationDO : materialSpecificationDOList) {
            map.put(materialSpecificationDO.getId(), materialSpecificationDO);
        }
        return map;
    }

    @Override
    public Boolean changeMsItemContentStatus(Long materialId) throws TuiaMediaException {
        return materialSpecificationItemContentDAO.changeStatus(materialId) > 0;
    }

    @Override
    public Map<Long, String> getMsImageUrl(List<Long> ids) throws TuiaMediaException {
        List<MaterialSpecificationItemContentDO> msList = materialSpecificationItemContentDAO.selectMsByIds(ids);
        Map<Long, String> map = new HashMap<>();
        for (MaterialSpecificationItemContentDO dto : msList) {
            if (StringUtils.contains(dto.getContent(), ",")) {
                map.put(dto.getId(), dto.getContent().split(",")[1]);
            }
        }
        return map;
    }

    @Override
    public List<Long> getMsIdByName(String msName) throws TuiaMediaException {

        return materialSpecificationDAO.selectMsIdByName(msName);
    }
}
