/**
 * Project Name:media-biz File Name:MediaAppConfigServiceImpl.java Package Name:cn.com.duiba.tuia.media.service.impl
 * Date:2016年10月8日下午3:38:59 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.ListUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSONObject;

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.api.dto.MediaAppConfigDto;
import cn.com.duiba.tuia.media.api.dto.MediaAppDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqCreateMediaAppConfigDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqGetMediaAppConfigByPageDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateMediaAppConfigDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAppInnerLogDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspBaseInnerLogDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMediaAppConfigDto;
import cn.com.duiba.tuia.media.dao.AccountDAO;
import cn.com.duiba.tuia.media.dao.BaseDAO;
import cn.com.duiba.tuia.media.dao.MediaAppConfigDao;
import cn.com.duiba.tuia.media.dao.MediaAppDao;
import cn.com.duiba.tuia.media.model.IdAndName;
import cn.com.duiba.tuia.media.service.MediaAppConfigService;

/**
 * ClassName:MediaAppConfigServiceImpl <br/>
 * Function: 媒体配置信息. <br/>
 * Date: 2016年10月8日 下午3:38:59 <br/>
 * 
 * @author guyan
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class MediaAppConfigServiceImpl implements MediaAppConfigService {

    /** logger */
    protected Logger            logger              = LoggerFactory.getLogger(MediaAppConfigServiceImpl.class);

    /** inner Log */
    protected Logger            innerLog            = LoggerFactory.getLogger("innerLog");

    @Autowired
    private MediaAppConfigDao   mediaAppCondfigDao;

    @Autowired
    private MediaAppDao         mediaAppDao;

    @Autowired
    private AccountDAO          accountDao;

    /** 数据类型-新增 */
    private static final String INSERT_DATA_TYPE    = "insert";

    /** 数据类型-修改 */
    private static final String UPDATE_DATA_TYPE    = "update";

    /** 时间格式 */
    private static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    @Override
    public Boolean createMediaAppConfig(ReqCreateMediaAppConfigDto param) throws TuiaMediaException {
        MediaAppConfigDto mediaAppConfigDto = new MediaAppConfigDto();
        mediaAppConfigDto.setAppId(param.getAppId());
        mediaAppConfigDto.setChargeType(param.getChargeType());
        mediaAppConfigDto.setPrice(param.getPrice());
        mediaAppConfigDto.setMediaSplitRatio(param.getMediaSplitRatio());
        mediaAppConfigDto.setPlatformSplitRatio(param.getPlatformSplitRatio());
        mediaAppConfigDto.setIsDeducted(param.getIsDeducted());
        mediaAppConfigDto.setDeductedPercent(param.getDeductedPercent());
        mediaAppCondfigDao.insert(mediaAppConfigDto);
        return true;
    }

    @SuppressWarnings("unchecked")
    @Override
    public PageResultDto<RspMediaAppConfigDto> getMediaAppCondfigList(ReqGetMediaAppConfigByPageDto mediaAppConfigQuery)
                                                                                                                        throws TuiaMediaException {
        mediaAppConfigQuery.setPageSize(ReqGetMediaAppConfigByPageDto.PAGE_SIZE_50);
        mediaAppConfigQuery.setRowStart((mediaAppConfigQuery.getCurrentPage() - 1) * mediaAppConfigQuery.getPageSize());
        mediaAppConfigQuery.setSort(ReqGetMediaAppConfigByPageDto.SORT_TYPE);
        mediaAppConfigQuery.setOrder(ReqGetMediaAppConfigByPageDto.ORDER_TYPE);
        List<MediaAppDto> mediaAppList;
        List<RspMediaAppConfigDto> rspMediaAppConfigList = null;
        List<Long> mediaIds;
        int totalNum = mediaAppCondfigDao.selectAmountByPageQuery(mediaAppConfigQuery);

        if (totalNum > 0 && totalNum >= mediaAppConfigQuery.getRowStart()) {
            mediaAppList = mediaAppCondfigDao.selectMediaAppConfigList(mediaAppConfigQuery);
            if (CollectionUtils.isNotEmpty(mediaAppList)) {
                rspMediaAppConfigList = new ArrayList<>(mediaAppList.size());
                mediaIds = new ArrayList<>(mediaAppList.size());
                for (MediaAppDto dto : mediaAppList) {
                    mediaIds.add(dto.getMediaId());
                }
                Map<Long, String> accountMap = getAccountMapByIds(mediaIds);
                for (MediaAppDto dto : mediaAppList) {
                    RspMediaAppConfigDto rspMediaAppConfig = new RspMediaAppConfigDto();
                    rspMediaAppConfig.setAppId(dto.getAppId());
                    rspMediaAppConfig.setAppName(dto.getAppName());
                    rspMediaAppConfig.setEmail(accountMap != null ? accountMap.get(dto.getMediaId()) : "");
                    rspMediaAppConfig.setCheckStatus(dto.getCheckStatus());
                    rspMediaAppConfig.setPlatform(dto.getPlatform());
                    rspMediaAppConfig.setCreateDate(new DateTime(dto.getGmtCreate()).toString("yyyy-MM-dd"));
                    rspMediaAppConfigList.add(rspMediaAppConfig);
                }
            } else {
                rspMediaAppConfigList = ListUtils.EMPTY_LIST;
            }
        }
        return new PageResultDto<>(totalNum, rspMediaAppConfigList, mediaAppConfigQuery.getPageSize());
    }

    @Override
    public MediaAppConfigDto getMediaAppConfigMsg(Long appId) throws TuiaMediaException {
        MediaAppDto mediaAppDto = this.mediaAppDao.selectMediaAppById(appId);
        if (mediaAppDto == null) {
            logger.error("the mediaAppDto is not exist, the appId = [{}]", appId);
            throw new TuiaMediaException(ErrorCode.E0401004);
        }
        MediaAppConfigDto mediaAppConfigDto = this.mediaAppCondfigDao.selectMediaAppConfigById(appId);
        if (mediaAppConfigDto == null) {
            mediaAppConfigDto = new MediaAppConfigDto();
            mediaAppConfigDto.setPlatformSplitRatio(MediaAppConfigDto.PLATFORM_SPLIT_RATIO_NUM);
            mediaAppConfigDto.setMediaSplitRatio(MediaAppConfigDto.MEDIA_SPLIT_RATIO_NUM);
            mediaAppConfigDto.setIsDeducted(MediaAppConfigDto.IS_DEDUCTED);
            mediaAppConfigDto.setDeductedPercent(MediaAppConfigDto.DEDUCTEDPERCENT_NUM);

        }
        mediaAppConfigDto.setAppId(mediaAppDto.getAppId());
        mediaAppConfigDto.setAppName(mediaAppDto.getAppName());
        return mediaAppConfigDto;
    }

    @Override
    public Boolean updateMediaAppConfig(ReqUpdateMediaAppConfigDto param) throws TuiaMediaException {
        MediaAppConfigDto mediaAppConfigDto = this.mediaAppCondfigDao.selectMediaAppConfigById(param.getAppId());
        if (mediaAppConfigDto == null) {
            Integer isDeducted = param.getIsDeducted() ? MediaAppConfigDto.IS_DEDUCTED : MediaAppConfigDto.NO_DEDUCTED;
            mediaAppConfigDto = new MediaAppConfigDto();
            mediaAppConfigDto.setAppId(param.getAppId());
            mediaAppConfigDto.setChargeType(param.getChargeType());
            mediaAppConfigDto.setDeductedPercent(param.getDeductedPercent());
            mediaAppConfigDto.setIsDeducted(isDeducted);
            mediaAppConfigDto.setMediaSplitRatio(param.getMediaSplitRatio());
            mediaAppConfigDto.setPlatformSplitRatio(param.getPlatformSplitRatio());
            if (param.getChargeType() != MediaAppConfigDto.CHARGE_TYPE_DIVIDED_INTO) {
                mediaAppConfigDto.setPrice(param.getPrice());
            }

            mediaAppConfigDto.setPrice(param.getPrice());
            Integer result = mediaAppCondfigDao.insert(mediaAppConfigDto);
            if (result > 0) {
                RspBaseInnerLogDto insertLogDto = buildAppConfigInnerLog(mediaAppConfigDto, INSERT_DATA_TYPE);
                innerLog.info(insertLogDto.toString());
            }
            return true;
        }
        if (param.getChargeType() == MediaAppConfigDto.CHARGE_TYPE_DIVIDED_INTO) {
            param.setPrice(mediaAppConfigDto.getPrice());
        } else {
            param.setMediaSplitRatio(mediaAppConfigDto.getMediaSplitRatio());
            param.setPlatformSplitRatio(mediaAppConfigDto.getPlatformSplitRatio());
        }
        Integer result = mediaAppCondfigDao.update(param);
        if (result > 0) {
            MediaAppConfigDto mediaAppConfigParam = mediaAppCondfigDao.selectMediaAppConfigById(param.getAppId());
            RspBaseInnerLogDto udpateLogDto = buildAppConfigInnerLog(mediaAppConfigParam, UPDATE_DATA_TYPE);
            innerLog.info(udpateLogDto.toString());
        }
        return true;
    }

    /**
     * 媒体用户Id List转换成map
     * 
     * @param mediaAppIds
     * @return
     * @throws TuiaMediaException
     */
    private Map<Long, String> getAccountMapByIds(List<Long> mediaIds) throws TuiaMediaException {
        if (CollectionUtils.isEmpty(mediaIds)) {
            return null;
        }

        int size = mediaIds.size();
        List<IdAndName> list = new ArrayList<>(size);
        Map<Long, String> accountMap = new HashMap<>(size);
        for (int i = 1, j = 0; i <= size; i++) {
            if (i % BaseDAO.BATCH_EXECUTE_MAX_LENGTH == 0 || i == size) {
                list.addAll(accountDao.selectMediaIdAndEmails(mediaIds.subList(j, i)));
                j = i;
            }
        }
        if (CollectionUtils.isNotEmpty(list)) {
            for (IdAndName accountVO : list) {
                // key:媒体用户 Id value:媒体用户 email
                accountMap.put(accountVO.getId(), accountVO.getName());
            }
        }
        return accountMap;
    }

    /**
     * buildAppInnerLog:(构建媒体配置信息内部日志数据). <br/>
     *
     * @return
     * @since JDK 1.6
     */
    private RspBaseInnerLogDto buildAppConfigInnerLog(MediaAppConfigDto appConfigDto, String type)
                                                                                                  throws TuiaMediaException {
        RspAppInnerLogDto dto = new RspAppInnerLogDto();
        MediaAppDto appDto = mediaAppDao.selectMediaAppById(appConfigDto.getAppId());
        if (appDto != null) {
            dto.setAppId(appDto.getAppId());
            dto.setAppName(appDto.getAppName());
            dto.setCheckStatus(appDto.getCheckStatus());
            dto.setMediaId(appDto.getMediaId());
            dto.setPlatform(appDto.getPlatform());
        }
        dto.setChargeType(appConfigDto.getChargeType());
        dto.setMediaSplitRatio((double)appConfigDto.getMediaSplitRatio()/100);
        dto.setPlatformSplitRatio((double)appConfigDto.getPlatformSplitRatio()/100);
        dto.setPrice(appConfigDto.getPrice());
        if (type.equals(INSERT_DATA_TYPE)) {
            dto.setTime(new DateTime(appConfigDto.getGmtCreate()).toString(YYYY_MM_DD_HH_MM_SS));
        } else if (type.equals(UPDATE_DATA_TYPE)) {
            dto.setTime(new DateTime(appConfigDto.getGmtModified()).toString(YYYY_MM_DD_HH_MM_SS));
        }
        return new RspBaseInnerLogDto(RspBaseInnerLogDto.GROUP_TYPE_LOG, RspBaseInnerLogDto.MEDIA_TYPE_LOG,
                                      new DateTime(new Date()).toString(YYYY_MM_DD_HH_MM_SS), null, null,
                                      JSONObject.toJSONString(dto));
    }
}
