/**
 * 文件名： AjaxUtils.java 此类描述的是： 作者: leiliang 创建时间: 2016年3月23日 上午10:46:40
 */
package cn.com.duiba.tuia.media.common.utils;

import java.io.IOException;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.com.duiba.tuia.media.common.constants.CommonConstants;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;

import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * <一句话功能描述> <功能详细描述>
 * 
 * @author: leiliang
 * @version:
 */
public class AjaxUtils {

    /** The logger. */
    private static Logger            logger           = LoggerFactory.getLogger(AjaxUtils.class);

    // -- header 常量定义 --//
    private static final String      HEADER_ENCODING  = "encoding";
    private static final String      HEADER_NOCACHE   = "no-cache";
    private static final String      DEFAULT_ENCODING = "UTF-8";
    private static final boolean     DEFAULT_NOCACHE  = true;

    public static final ObjectMapper MAPPER           = JsonUtils.mapper;

    static {
        MAPPER.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
    }

    private AjaxUtils                au;

    public AjaxUtils getAjaxUtils() {
        if (au == null) {
            au = new AjaxUtils();
        }
        return au;
    }

    /**
     * render:(直接输出内容的简便函数). <br/>
     *
     * @author ZFZ
     * @param response
     * @param contentType
     * @param content
     * @param headers 可变的header数组，目前接受的值为"encoding:"或"no-cache:",默认值分别为UTF-8和true.
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void render(HttpServletResponse response, final String contentType, final String content,
                              final String... headers) throws TuiaMediaException {
        initResponseHeader(response, contentType, headers);
        PrintWriter write = null;
        try {
            write = response.getWriter();
            write.write(content);
            write.flush();
        } catch (IOException e) {
            logger.error("AjaxUtils.render happen error! the contentType=[{}],the content=[{}],the headers=[{}], the e=[{}]",
                         contentType, content, headers, e);
            throw new TuiaMediaException(ErrorCode.E0002006);
        } finally {
            if (write != null) {
                write.close();
            }
        }
    }

    /**
     * renderText:(直接输出文本.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param text
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderText(HttpServletResponse response, final String text, final String... headers)
                                                                                                           throws TuiaMediaException {
        render(response, CommonConstants.ContentType.TEXT_TYPE, text, headers);
    }

    /**
     * renderHtml:(直接输出HTML.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param html
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderHtml(HttpServletResponse response, final String html, final String... headers)
                                                                                                           throws TuiaMediaException {
        render(response, CommonConstants.ContentType.HTML_TYPE, html, headers);
    }

    /**
     * renderXml:(直接输出XML.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param xml
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderXml(HttpServletResponse response, final String xml, final String... headers)
                                                                                                         throws TuiaMediaException {
        render(response, CommonConstants.ContentType.XML_TYPE, xml, headers);
    }

    /**
     * renderJson:(直接输出JSON.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param jsonString
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderJson(HttpServletResponse response, final String jsonString, final String... headers)
                                                                                                                 throws TuiaMediaException {
        render(response, CommonConstants.ContentType.JSON_TYPE, jsonString, headers);
    }

    /**
     * renderJson:(直接输出JSON,使用Jackson转换Java对象.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param data 可以是List<POJO>, POJO[], POJO, 也可以Map名值对.
     * @param headers
     * @since JDK 1.6
     */
    public static void renderJson(HttpServletResponse response, final Object data, final String... headers) {
        initResponseHeader(response, CommonConstants.ContentType.JSON_TYPE, headers);
        try {
            MAPPER.writeValue(response.getWriter(), data);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * renderJsonp:(直接输出支持跨域Mashup的JSONP.). <br/>
     *
     * @author ZFZ
     * @param response
     * @param callbackName callback函数名.
     * @param object Java对象,可以是List<POJO>, POJO[], POJO ,也可以Map名值对, 将被转化为json字符串.
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderJsonp(HttpServletResponse response, final String callbackName, final Object object,
                                   final String... headers) throws TuiaMediaException {
        String jsonString = null;
        try {
            jsonString = MAPPER.writeValueAsString(object);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }

        String result = new StringBuilder().append(callbackName).append("(").append(jsonString).append(");").toString();

        /** 渲染Content-Type为javascript的返回内容,输出结果为javascript语句, 如callback197("{html:'Hello World!!!'}"); */
        render(response, CommonConstants.ContentType.JS_TYPE, result, headers);
    }

    /**
     * 分析并设置contentType与headers.
     */
    private static void initResponseHeader(HttpServletResponse response, final String contentType,
                                           final String... headers) {
        // 分析headers参数
        String encoding = DEFAULT_ENCODING;
        boolean noCache = DEFAULT_NOCACHE;
        for (String header : headers) {
            String headerName = StringUtils.substringBefore(header, ":");
            String headerValue = StringUtils.substringAfter(header, ":");

            if (StringUtils.equalsIgnoreCase(headerName, HEADER_ENCODING)) {
                encoding = headerValue;
            } else if (StringUtils.equalsIgnoreCase(headerName, HEADER_NOCACHE)) {
                noCache = Boolean.parseBoolean(headerValue);
            } else {
                throw new IllegalArgumentException(headerName + "不是一个合法的header类型");
            }
        }

        // 设置headers参数
        String fullContentType = contentType + ";charset=" + encoding;
        response.setContentType(fullContentType);
        if (noCache) {
            setDisableCacheHeader(response);
        }
    }

    /**
     * 设置禁止客户端缓存的Header.
     */
    public static void setDisableCacheHeader(HttpServletResponse response) {
        // Http 1.0 header
        response.setDateHeader("Expires", 1L);
        response.addHeader("Pragma", "no-cache");
        // Http 1.1 header
        response.setHeader("Cache-Control", "no-cache, no-store, max-age=0");
    }

    /**
     * renderJsonporJson:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param response
     * @param callbackName
     * @param object
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderJsonporJson(HttpServletResponse response, final String callbackName, final Object object,
                                         final String... headers) throws TuiaMediaException {
        if (callbackName != null && !"".equals(callbackName)) {
            renderJsonp(response, callbackName, object, headers);
        } else {
            renderJson(response, object, headers);
        }
    }

    /**
     * renderJsonporJson2:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param response
     * @param callbackName
     * @param data
     * @param headers
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static void renderJsonporJson2(HttpServletResponse response, final String callbackName, final String data,
                                          final String... headers) throws TuiaMediaException {
        if (callbackName != null && !"".equals(callbackName)) {

            String result = callbackName + "(" + data + ")";
            renderText(response, result, headers);
        } else {
            renderJson(response, data, headers);
        }
    }
}
