package cn.com.duiba.tuia.media.common.tool;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;

import com.aliyun.api.AliyunClient;
import com.aliyun.api.DefaultAliyunClient;
import com.aliyun.api.cdn.cdn20141111.request.RefreshObjectCachesRequest;
import com.aliyun.api.cdn.cdn20141111.response.RefreshObjectCachesResponse;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.ObjectMetadata;
import com.taobao.api.ApiException;

/**
 * The Class UploadTool.
 */
public class UploadTool {

    /** The log. */
    private static Logger log = LoggerFactory.getLogger(UploadTool.class);

    /** The access id. */
    private static String accessId;

    /** The access key. */
    private static String accessKey;

    /** The end point. */
    private static String endPoint;

    /** The bucket name. */
    private static String bucketName;

    /** The server url. */
    private static String serverUrl;

    /** The object path. */
    private static String objectPath;

    /** The tuia url. */
    private static String tuiaUrl;

    /**
     * 上传Oss文件.
     *
     * @param file the file
     * @param objectName the object name
     * @return true, if upload oss
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOss(File file, String objectName) throws TuiaMediaException {
        uploadOss(file, objectName, null);
    }

    /**
     * oss上传文件.
     *
     * @param file 文件
     * @param objectName 存储路径
     * @param contentType 文件类型
     * @throws TuiaMediaException the tuia exception
     */
    private static void uploadOss(File file, String objectName, String contentType) throws TuiaMediaException {//NOSONAR
        OSSClient clientOss = new OSSClient(endPoint, accessId, accessKey);
        // 输出结果
        if (clientOss.doesBucketExist(bucketName)) {
            try (InputStream content = new FileInputStream(file)) {
                // 创建上传Object的Metadata
                ObjectMetadata meta = new ObjectMetadata();
                // 必须设置ContentLength
                meta.setContentLength(file.length());
                if (null != contentType) {
                    meta.setContentType(contentType);
                }
                boolean flag = false;
                if (isFileExist(objectName)) {
                    flag = true;
                }
                // 上传Object.
                clientOss.putObject(bucketName, objectName, content, meta);
                if (flag) {
                    refreshObjectCaches(objectName);
                }
            } catch (OSSException e) {
                log.error("oss upload happen OSSException!", e);
                throw new TuiaMediaException(ErrorCode.E0003003, e);
            } catch (ClientException e) {
                log.error("oss upload happen ClientException!", e);
                throw new TuiaMediaException(ErrorCode.E0003003, e);
            } catch (FileNotFoundException e) {
                log.error("oss upload file error! beacuse the file[{}] not found", file, e);
                throw new TuiaMediaException(ErrorCode.E0003003, e);
            } catch (IOException e) {
                log.error("oss upload file happen IOException", e);
            }
        }
    }

    /**
     * 上传OssCss文件.
     *
     * @param file the file
     * @param objectName the object name
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOssCss(File file, String objectName) throws TuiaMediaException {
        uploadOss(file, objectName, "text/css");
    }

    /**
     * 上传OssJs文件.
     *
     * @param file the file
     * @param objectName the object name
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOssJs(File file, String objectName) throws TuiaMediaException {
        uploadOss(file, objectName, "text/js");
    }

    /**
     * 上传OssImg文件.
     *
     * @param file the file
     * @param objectName the object name
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOssImg(File file, String objectName) throws TuiaMediaException {
        uploadOss(file, objectName, "image/jpeg");
    }

    /**
     * 上传OssImg文件.
     *
     * @param content the content
     * @param objectName the object name
     * @param size the size
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOssImg(InputStream content, String objectName, long size) throws TuiaMediaException {
        uploadOss(content, objectName, size, "image/jpeg");
    }

    /**
     * Upload oss.
     *
     * @param content the content
     * @param objectName the object name
     * @param size the size
     * @throws TuiaMediaException the tuia exception
     */
    public static void uploadOss(InputStream content, String objectName, long size) throws TuiaMediaException {
        uploadOss(content, objectName, size, null);
    }

    /**
     * Upload oss.
     *
     * @param content the content
     * @param objectName the object name
     * @param size the size
     * @param contentType the content type
     * @throws TuiaMediaException the tuia exception
     */
    private static void uploadOss(InputStream content, String objectName, long size, String contentType)
                                                                                                        throws TuiaMediaException {
        OSSClient clientOss = new OSSClient(endPoint, accessId, accessKey);
        // 输出结果
        if (clientOss.doesBucketExist(bucketName)) {
            try {
                // 创建上传Object的Metadata
                ObjectMetadata meta = new ObjectMetadata();
                // 必须设置ContentLength
                meta.setContentLength(size);
                if (null != contentType) {
                    meta.setContentType(contentType);
                }

                boolean flag = false;
                if (isFileExist(objectName)) {
                    flag = true;
                }
                // 上传Object.
                clientOss.putObject(bucketName, objectName, content, meta);
                if (flag) {
                    refreshObjectCaches(objectName);
                }
            } catch (OSSException e) {
                log.error("oss upload file happen OSSException!", e);
                throw new TuiaMediaException(ErrorCode.E0003003, e);
            } catch (ClientException e) {
                log.error("oss upload file happen ClientException!", e);
                throw new TuiaMediaException(ErrorCode.E0003003, e);
            }
        }
    }

    /**
     * CDN刷新缓存.
     *
     * @param path the path
     * @throws TuiaMediaException the tuia exception
     */
    public static void refreshObjectCaches(String path) throws TuiaMediaException {
        AliyunClient clientCdn = new DefaultAliyunClient(serverUrl, accessId, accessKey);
        RefreshObjectCachesRequest request = new RefreshObjectCachesRequest();
        request.setObjectType(ObjectType.File.name());
        request.setObjectPath(objectPath + path);
        try {
            RefreshObjectCachesResponse response = clientCdn.execute(request);
            if (!StringUtils.isEmpty(response.getErrorCode())) {
                log.error("refresh caches error! the path=[{}] and the message=[{}]", path, response.getMessage());
                throw new TuiaMediaException(ErrorCode.E0003003);
            }
        } catch (ApiException e) {
            log.error("refresh caches happen ApiException! the path is:" + path, e);
            throw new TuiaMediaException(ErrorCode.E0003003, e);
        }
    }

    /**
     * 验证文件是否存在.
     *
     * @param objectName the object name
     * @return true, if checks if is file exist
     */
    public static boolean isFileExist(String objectName) {
        OSSClient clientOss = new OSSClient(endPoint, accessId, accessKey);
        try {
            clientOss.getObject(bucketName, objectName);
            return true;
        } catch (ClientException e) {
            log.error("oss upload file happen ClientException", e);
            return false;
        } catch (OSSException e) {
            log.error("oss upload file happen OSSException", e);
            return false;
        }

    }

    /**
     * 刷新路径类型.
     */
    enum ObjectType {
        /* 文件 */
        /** The File. */
        File,
        /* 目录 */
        /** The Directory. */
        Directory;
    }

    /**
     * Delete object.
     *
     * @param objectName the object name
     */
    public void deleteObject(String objectName) {
        OSSClient clientOss = new OSSClient(endPoint, accessId, accessKey);
        // 删除Object
        try {
            clientOss.deleteObject(bucketName, objectName);
        } catch (OSSException e) {
            log.error("deleteObject happen OSSException! the objectName=[{}]", objectName, e);
        } catch (ClientException e) {
            log.error("deleteObject happen ClientException! the objectName=[{}]", objectName, e);
        }
    }

    /**
     * Sets the access id.
     *
     * @param accessId the access id
     */
    public static void setAccessId(String accessId) {
        UploadTool.accessId = accessId;
    }

    /**
     * Sets the access key.
     *
     * @param accessKey the access key
     */
    public static void setAccessKey(String accessKey) {
        UploadTool.accessKey = accessKey;
    }

    /**
     * Sets the end point.
     *
     * @param endPoint the end point
     */
    public static void setEndPoint(String endPoint) {
        UploadTool.endPoint = endPoint;
    }

    /**
     * Sets the bucket name.
     *
     * @param bucketName the bucket name
     */
    public static void setBucketName(String bucketName) {
        UploadTool.bucketName = bucketName;
    }

    /**
     * Sets the server url.
     *
     * @param serverUrl the server url
     */
    public static void setServerUrl(String serverUrl) {
        UploadTool.serverUrl = serverUrl;
    }

    /**
     * Sets the object path.
     *
     * @param objectPath the object path
     */
    public static void setObjectPath(String objectPath) {
        UploadTool.objectPath = objectPath;
    }

    /**
     * Gets the tuia url.
     *
     * @return the tuiaUrl
     */
    public static String getTuiaUrl() {
        return tuiaUrl;
    }

    /**
     * Sets the tuia url.
     *
     * @param tuiaUrl the tuiaUrl to set
     */
    public static void setTuiaUrl(String tuiaUrl) {
        UploadTool.tuiaUrl = tuiaUrl;
    }
}
