/**
 * Project Name:media-deploy<br>
 * File Name:TestController.java<br>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br>
 * Date:2016年10月12日下午4:59:34<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.ApiOperation;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import cn.com.duiba.tuia.media.api.dto.MediaAppDataDto;
import cn.com.duiba.tuia.media.api.dto.SlotCacheDto;
import cn.com.duiba.tuia.media.api.dto.StrategyDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.remoteservice.RemoteMediaServiceImpl;
import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * ClassName: TestController <br/>
 * Function: 测试. <br/>
 * date: 2016年10月12日 下午4:59:34 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
// @Controller
// @RequestMapping("/remoteTest")
// @Api("dubbo测试相关api")
public class RemoteTestController {

    @Autowired
    private RemoteMediaServiceImpl remoteMediaServiceImpl;

    private <T> DubboResult<T> failResult(Exception e) {
        if (e instanceof TuiaMediaException) {
            TuiaMediaException e1 = (TuiaMediaException) e;
            return DubboResult.failResult(e1.getResultCode(), e1.getMessage());
        } else {
            return DubboResult.failResult(ErrorCode.E9999999.getErrorCode(), "未知错误");
        }
    }

    /**
     * 查询屏蔽策略信息.<br>
     * [使用说明]
     * <ol>
     * <li>当发生数据库失败时：returnCode=0002001</li>
     * <li>当该广告位不存在时：returnCode=0304008</li>
     * <li>当该屏蔽策略不存在时：returnCode=0304002</li>
     * <li>当发生其他错误时：returnCode=9999999</li>
     * </ol>
     *
     * @param slotId the slot id
     * @return the strategy
     */
    @ResponseBody
    @ApiOperation(value = "查询屏蔽策略", httpMethod = "GET", notes = "查询屏蔽策略接口")
    @RequestMapping(value = "/getStrategy", method = RequestMethod.GET)
    DubboResult<StrategyDto> getStrategy(@RequestParam long slotId) {
        try {
            return remoteMediaServiceImpl.getStrategy(slotId);
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 查询媒体APP是否有效.<br>
     * [使用说明]<br>
     * true:有效<br>
     * false:无效<br>
     *
     * @param mediaAppId 媒体APPid
     * @return the dubbo result< boolean>
     */
    @ResponseBody
    @ApiOperation(value = "判断媒体应用是否有效", httpMethod = "GET", notes = "判断媒体应用是否有效接口")
    @RequestMapping(value = "/isValidMediaApp", method = RequestMethod.GET)
    DubboResult<Boolean> isValidMediaApp(@RequestParam long mediaAppId) {
        try {
            return remoteMediaServiceImpl.isValidMediaApp(mediaAppId);
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 查询媒体应用信息.<br>
     * [使用说明]
     * <ol>
     * <li>当发生数据库失败时：returnCode=0002001</li>
     * <li>当该媒体应用不存在时：returnCode=0401001</li>
     * <li>当发生其他错误时：returnCode=9999999</li>
     * </ol>
     * 
     * @param mediaAppId 媒体应用ID
     * @return the media app
     */
    @ResponseBody
    @ApiOperation(value = "查询媒体应用", httpMethod = "GET", notes = "查询媒体应用接口")
    @RequestMapping(value = "/getMediaApp", method = RequestMethod.GET)
    DubboResult<MediaAppDataDto> getMediaApp(@RequestParam long mediaAppId) {
        try {
            return remoteMediaServiceImpl.getMediaApp(mediaAppId);
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 查询广告位信息.<br>
     * [使用说明]
     * <ol>
     * <li>当发生数据库失败时：returnCode=0002001</li>
     * <li>当该广告位不存在时：returnCode=0304008</li>
     * <li>当发生其他错误时：returnCode=9999999</li>
     * </ol>
     *
     * @param slotId the slot id
     * @return the dubbo result< boolean>
     */
    @ResponseBody
    @ApiOperation(value = "查询广告位", httpMethod = "GET", notes = "查询广告位接口")
    @RequestMapping(value = "/getSlot", method = RequestMethod.GET)
    DubboResult<SlotCacheDto> getSlot(@RequestParam long slotId) {
        try {
            return remoteMediaServiceImpl.getSlot(slotId);
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 更新广告位的屏蔽策略缓存.<br>
     * [使用说明]
     * <ol>
     * <li>当发生数据库失败时：returnCode=0002001</li>
     * <li>当该广告位不存在时：returnCode=0304008</li>
     * <li>当该屏蔽策略不存在时：returnCode=0304002</li>
     * <li>当发生其他错误时：returnCode=9999999</li>
     * </ol>
     *
     * @param slotId 广告位id
     * @return the dubbo result< strategy dto>
     */
    @ResponseBody
    @ApiOperation(value = "更新广告位屏蔽策略缓存", httpMethod = "POST", notes = "更新广告位屏蔽策略缓存接口")
    @RequestMapping(value = "/updateStrategy", method = RequestMethod.POST)
    DubboResult<StrategyDto> updateStrategy(@RequestParam long slotId) {
        try {
            return remoteMediaServiceImpl.updateStrategy(slotId);
        } catch (Exception e) {
            return failResult(e);
        }
    }
}
