/**
 * Project Name:media-deploy<br/>
 * File Name:AbsenseController.java<br/>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br/>
 * Date:2016年9月26日下午5:02:40<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;

import javax.validation.Valid;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateSlot;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotDetailDto;
import cn.com.duiba.tuia.media.bo.SlotBackendBo;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.constants.SlotPictureSizeEnmu;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqCreateSlot;
import cn.com.duiba.tuia.media.model.req.ReqEditSlotEnableStatus;
import cn.com.duiba.tuia.media.model.req.ReqSlotList;
import cn.com.duiba.tuia.media.model.rsp.RspPageResult;
import cn.com.duiba.tuia.media.model.rsp.RspSlot;
import cn.com.duiba.tuia.media.service.SlotService;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.web.aop.LogWrite;

/**
 * 广告位相关controller <br/>
 * Date: 2016年9月26日 下午5:02:40 <br/>
 * .
 *
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */

@Controller
@RequestMapping("/slot")
@Api("广告位相关接口")
public class SlotController extends BaseController {

    /** The slot service. */
    @Autowired
    private SlotService   slotService;

    @Autowired
    private SlotBackendBo slotBackendBo;

    /**
     * 创建广告位.
     *
     * @param req the req
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "创建广告位")
    @ResponseBody
    @ApiOperation(value = "创建广告位", httpMethod = "POST", notes = "创建广告位接口")
    @RequestMapping(value = "/create", method = RequestMethod.POST)
    public Result<Boolean> createSlot(@Valid @RequestBody ReqCreateSlot req, BindingResult result) {
        try {
            checkParam(result);
            RequestLocal rl = RequestLocal.get();
            req.setMediaId(rl.getCid());

            // 如果是信息流 则必须有图片尺寸和元素
            if (SlotDto.ADSENSE_TYPE_INFORMATION_STREAM == req.getSlotType()) {
                if (req.getPictureSize() == null || CollectionUtils.isEmpty(req.getElementType())) {
                    throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "信息流,必须有图片尺寸和元素");
                }
            } else if (SlotDto.ADSENSE_TYPE_PLAQUE == req.getSlotType()) {
                req.setPictureSize(SlotPictureSizeEnmu.SIZE_41.getSizeId());
            } else if (SlotDto.ADSENSE_TYPE_STREAMER_BANNER == req.getSlotType()) {
                req.setPictureSize(SlotPictureSizeEnmu.SIZE_61.getSizeId());
            } else if (SlotDto.ADSENSE_TYPE_BANNER == req.getSlotType()) {
                req.setPictureSize(SlotPictureSizeEnmu.SIZE_81.getSizeId());
            } else if (SlotDto.ADSENSE_TYPE_BUOY == req.getSlotType()) {
                req.setPictureSize(SlotPictureSizeEnmu.SIZE_21.getSizeId());
            } else if (SlotDto.ADSENSE_TYPE_APP_WALL == req.getSlotType()) {
                req.setPictureSize(SlotPictureSizeEnmu.SIZE_21.getSizeId());
            } else {
                throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "广告位类型未知");
            }

            return successResult(slotService.insert(req));
        } catch (Exception e) {
            logger.error("SlotController.createSlot error! the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * Gets the list.
     *
     * @param param the param
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the list
     */
    @ResponseBody
    @ApiOperation(value = "查询广告位列表", httpMethod = "GET", notes = "查询广告位列表")
    @RequestMapping(value = "/getList", method = RequestMethod.GET)
    public Result<RspPageResult<RspSlot>> getList(@ModelAttribute ReqSlotList param) {
        try {
            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            param.setPageSize(ReqSlotList.PAGE_SIZE_50);
            param.setRowStart((param.getCurrentPage() - 1) * param.getPageSize());
            return successResult(slotService.selectListByPage(param));
        } catch (Exception e) {
            logger.error("SlotController.getList error! the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 查看广告位详情.
     *
     * @param slotId the slot id
     * @return the detail
     */
    @ApiOperation(value = "查看广告位详情", httpMethod = "GET", notes = "查看广告位详情接口")
    @ApiImplicitParam(name = "slotId", value = "广告位ID", required = true, dataType = "Long", paramType = "query")
    @ResponseBody
    @RequestMapping("/getDetail")
    public Result<RspSlotDetailDto> getDetail(@RequestParam Long slotId) {
        try {
            if (slotId == null) {
                logger.error("slotIds can not be null");
                throw new TuiaMediaException(ErrorCode.E0001005);
            }
            return successResult(slotBackendBo.getSlotDetailById(slotId));
        } catch (Exception e) {
            logger.error("slotController getDetail error! the slotId=[{}]", slotId);
            return failResult(e);
        }
    }

    /**
     * 编辑广告位.
     *
     * @param param the param
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "编辑广告位")
    @ResponseBody
    @ApiOperation(value = "编辑广告位", httpMethod = "POST", notes = "编辑广告位")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result<Boolean> update(@Valid @RequestBody ReqUpdateSlot param, BindingResult result) {
        try {
            checkParam(result);

            // 如果是信息流 则必须有图片尺寸和元素
            if (SlotDto.ADSENSE_TYPE_INFORMATION_STREAM == param.getSlotType()
                && (param.getPictureSize() == null || param.getElementType() == null)) {
                throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "信息流,必须有图片尺寸和元素");
            }

            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            return successResult(slotService.update(param));
        } catch (Exception e) {
            logger.error("SlotController.update error! the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 广告位开启/关闭.
     *
     * @param param the param
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "广告位开启/关闭")
    @ResponseBody
    @ApiOperation(value = "广告位开启/关闭", httpMethod = "POST", notes = "广告位开启/关闭")
    @RequestMapping(value = "/updateEnableStatus", method = RequestMethod.POST)
    public Result<Boolean> updateEnableStatus(@Valid @RequestBody ReqEditSlotEnableStatus param, BindingResult result) {
        try {
            checkParam(result);
            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            return successResult(slotService.updateEnableStatus(param));
        } catch (Exception e) {
            logger.error("SlotController.updateEnableStatus error! the param=[{}]", param);
            return failResult(e);
        }
    }

}
