package cn.com.duiba.tuia.media.web.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.common.constants.CommonConstants.COOKIE_KEY;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.RequestTool;
import cn.com.duiba.tuia.media.common.tool.TimeProfileTool;
import cn.com.duiba.tuia.media.common.utils.AjaxUtils;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.service.BaseService;
import cn.com.duiba.tuia.media.utils.CookieUtil;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.utils.ResultUtil;

/**
 * The Class LoginFilter.
 */
@Component("loginFilter")
public class LoginFilter extends BaseService implements Filter {

    /** The log. */
    private Logger             log              = LoggerFactory.getLogger(LoginFilter.class);

    /** 权限过了列表(不需要做权限校验). */
    public static List<String> AUTH_FILTER_LIST = new ArrayList<String>();

    @Autowired
    private AccountService     accountService;

    /**
     * Do filter.
     *
     * @param req the req
     * @param resp the resp
     * @param chain the chain
     * @throws IOException the IO exception
     * @throws ServletException the servlet exception
     */
    @Override
    public void doFilter(ServletRequest req, ServletResponse resp, FilterChain chain) throws IOException,
                                                                                     ServletException {
        TimeProfileTool.enter("loginFilter.doFilter");

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) resp;
        RequestTool.setRequestInThreadLocal(request);

        RequestLocal.clear();
        RequestLocal.get().setRequest(request);
        RequestLocal.get().setResponse(response);

        Long cid = RequestLocal.get().getCid();
        String path = request.getServletPath();

        if (path.startsWith("/webjars") || path.startsWith("/swagger") || AUTH_FILTER_LIST.contains(path)) {
            // 免拦截页面或请求
        } else if (cid != null) {

            if ("/account/login".equals(path) || "/account/register".equals(path)) {
                // 登录过且未失效, 再重新登录的话重定向到首页
                log.error("please enter the home page! the url is:" + path);
                response.sendRedirect("/#/private");
                return;
            }

            // 校验用户状态
            try {
                accountStatusAuth(cid);
            } catch (TuiaMediaException e) {
                // 清除cookie
                CookieUtil.deleteCookie(COOKIE_KEY.ACCOUNT_KEY);
                log.error("please login again! the url is:" + path);
                exceptionFailure(response, e);
                return;
            }

        } else if ("/private".equals(path)) {
            response.sendRedirect("/#/signin");
            return;
        } else if (!AUTH_FILTER_LIST.contains(path)) {
            log.error("please login again! the url is:" + path);
            AjaxUtils.renderJson(response, ResultUtil.fail(ErrorCode.E9999998));
            return;
        }

        chain.doFilter(req, resp);
        TimeProfileTool.release();
    }

    /**
     * 校验账号状态.
     *
     * @param accountId the account id
     * @param auditStatus 账号审核状态
     * @param freezeStatus 冻结状态
     * @throws TuiaException 失败抛出异常
     */
    private void accountStatusAuth(long mediaId) throws TuiaMediaException {
        AccountDto accountDto = accountService.selectByIdNotNull(mediaId);
        doCheckAudit(mediaId, accountDto.getCheckStatus());
        doCheckFreeze(mediaId, accountDto.getFreezeStatus());
    }

    /**
     * 校验账户是否冻结. <br/>
     *
     * @author xiawei
     * @param accountId
     * @param freezeStatus
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private void doCheckFreeze(long mediaId, Integer freezeStatus) throws TuiaMediaException {
        if (AccountDto.FREEZED_STATUS == freezeStatus) {
            // 被冻结状态
            logger.error("the account is freeze, the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102004);
        }
        if (AccountDto.UNFREEZED_STATUS != freezeStatus) {
            // 非法的冻结状态(既不是冻结也不是未冻结)
            logger.error("the account freeze status is illegal,  the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102005);
        }
    }

    /**
     * 校验账户是否审核. <br/>
     *
     * @author xiawei
     * @param accountId
     * @param auditStatus
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private void doCheckAudit(long mediaId, Integer auditStatus) throws TuiaMediaException {
        if (AccountDto.CHECK_STATUS_ING == auditStatus) {
            // 审核中状态
            logger.error("user account in review status, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102006);
        }
        if (AccountDto.CHECK_STATUS_REFUSE == auditStatus) {
            // 被拒绝状态
            logger.error("user account in refuse status, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102007);
        }
        if (AccountDto.CHECK_STATUS_PASS != auditStatus) {
            // 非法的审核状态
            logger.error("the account status is illegal, the audit status=[{}]", auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102008);
        }
    }

    public void exceptionFailure(HttpServletResponse response, Exception e) {
        if (e instanceof TuiaMediaException) {
            AjaxUtils.renderJson(response,
                                 ResultUtil.fail(((TuiaMediaException) e).getResultCode(),
                                                 ((TuiaMediaException) e).getResultMessage()));
        } else {
            log.error("系统错误", e);
            AjaxUtils.renderJson(response, ResultUtil.fail(ErrorCode.E9999999));
        }
    }

    /**
     * Destroy.
     */
    @Override
    public void destroy() {

    }

    /**
     * Inits the.
     *
     * @param fc the fc
     * @throws ServletException the servlet exception
     */
    @Override
    public void init(FilterConfig fc) throws ServletException {
        // 前台路径
        AUTH_FILTER_LIST.add("/");
        AUTH_FILTER_LIST.add("/private");
        AUTH_FILTER_LIST.add("/index.html");
        AUTH_FILTER_LIST.add("/favicon.ico");
        AUTH_FILTER_LIST.add("/#/signin");
        AUTH_FILTER_LIST.add("/private#/404");
        AUTH_FILTER_LIST.add("/__webpack_hmr");
        // swagger
        AUTH_FILTER_LIST.add("/swagger-ui.html");
        AUTH_FILTER_LIST.add("/images/favicon-16x16.png");
        AUTH_FILTER_LIST.add("/swagger-resources");
        AUTH_FILTER_LIST.add("/v2/api-docs");
        AUTH_FILTER_LIST.add("/images/favicon-32x32.png");
        AUTH_FILTER_LIST.add("/images/favicon-16x16.png");
        AUTH_FILTER_LIST.add("/configuration/security");
        AUTH_FILTER_LIST.add("/configuration/ui");

        // 上传文件接口
        AUTH_FILTER_LIST.add("/upload/index");

        // 用户相关接口
        AUTH_FILTER_LIST.add("/account/resetPasswd/redirect");
        AUTH_FILTER_LIST.add("/account/sendResetPasswdEmail");
        AUTH_FILTER_LIST.add("/account/resetPasswd");
        AUTH_FILTER_LIST.add("/account/isExists");
        AUTH_FILTER_LIST.add("/account/isExistPhone");
        AUTH_FILTER_LIST.add("/account/register");
        AUTH_FILTER_LIST.add("/account/login");
        AUTH_FILTER_LIST.add("/account/verifyEmail");
        AUTH_FILTER_LIST.add("/account/repeatVerifyEmail");
        AUTH_FILTER_LIST.add("/account/updateAuditData");
        AUTH_FILTER_LIST.add("/account/getUncheckAccount");
        AUTH_FILTER_LIST.add("/account/sendResetPdEmail");
        AUTH_FILTER_LIST.add("/account/verifyResetPdCode");
        AUTH_FILTER_LIST.add("/account/resetPassword");
        AUTH_FILTER_LIST.add("/account/logout");

        // 短信相关接口
        AUTH_FILTER_LIST.add("/sms/send");
        AUTH_FILTER_LIST.add("/sms/verify");
    }

}
