/**
 * Project Name:media-deploy<br/>
 * File Name:AccountChangeController.java<br/>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br/>
 * Date:2016年12月6日下午9:16:42<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import cn.com.duiba.tuia.media.api.dto.req.ReqAccountChange;
import cn.com.duiba.tuia.media.api.dto.rsp.AccountChangeRsp;
import cn.com.duiba.tuia.media.api.dto.rsp.RspAccountChangeResult;
import cn.com.duiba.tuia.media.api.remoteservice.RemoteAccountChangeBackendService;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqSlotList;
import cn.com.duiba.tuia.media.service.AccountChangeService;
import cn.com.duiba.tuia.media.utils.RequestLocal;

/**
 * ClassName:AccountChangeController <br/>
 * Date: 2016年12月6日 下午9:16:42 <br/>
 * 
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */

@Controller
@RequestMapping("/accountChange")
@Api("账户资金流水相关api")
public class AccountChangeController extends BaseController {

    /** The Constant YYYY_MM_DD. */
    private static final String               YYYY_MM = "yyyy-MM";

    @Autowired
    private AccountChangeService              accountChangeService;

    @Autowired
    private RemoteAccountChangeBackendService remoteAccountChangeBackendService;

    /**
     * getAccoutChangeList:(分页查询资金流水列表). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "分页查询资金流水列表", httpMethod = "GET", notes = "查询资金流水列表")
    @ResponseBody
    @RequestMapping(value = "/getAccoutChangeList", method = RequestMethod.GET)
    public Result<RspAccountChangeResult<AccountChangeRsp>> getAccoutChangeList(@Valid @ModelAttribute ReqAccountChange req,
                                                                                BindingResult result) {
        try {
            checkParam(result);
            Long cid = RequestLocal.get().getCid();
            req.setEndDate(new DateTime(req.getEndDate()).millisOfDay().withMaximumValue().toString("yyyy-MM-dd HH:mm:ss"));
            req.setMediaId(cid);
            req.setPageSize(req.getPageSize() == null ? ReqSlotList.PAGE_SIZE_50 : req.getPageSize());
            req.setRowStart((req.getCurrentPage() - 1) * req.getPageSize());
            return successResult(accountChangeService.selectAccountChangeByPage(req));
        } catch (Exception e) {
            logger.error("AccountChangeController.getAccoutChangeList error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * exportReport:(资金流水报表接口). <br/>
     *
     * @author ZFZ
     * @param req
     * @param response
     * @since JDK 1.6
     */
    @ApiOperation(value = "资金流水报表导出接口", httpMethod = "GET")
    @ResponseBody
    @RequestMapping(value = "/exportReport", method = RequestMethod.GET)
    public void exportReport(@Valid @ModelAttribute ReqAccountChange req, HttpServletResponse response) {

        OutputStream out = null;
        try {
            Long cid = RequestLocal.get().getCid();
            req.setMediaId(cid);
            req.setPageSize(-1);
            // 设置响应头和保存文件名
            setExportResponse(buildExportFile(new DateTime(req.getStartDate()).toString(YYYY_MM),
                                              "-" + new DateTime(req.getEndDate()).toString(YYYY_MM) + "_资金流水报表数据"),
                              response);
            // 通过response.outputStream响应告诉我们，这个文档不是保存到服务器端，而是直接发送给浏览器
            out = response.getOutputStream();
            req.setEndDate(new DateTime(req.getEndDate()).millisOfDay().withMaximumValue().toString("yyyy-MM-dd HH:mm:ss"));
            accountChangeService.exportAccountChange(req, out);
            successResult(true);
        } catch (Exception e) {
            logger.error("AccountChangeController.exportReport error!,the req=[{}]", req);
            failResult(e);
        } finally {
            closeOutputStream(out);
        }
    }

    /**
     * Builds the export file.
     *
     * @param startDate the start date
     * @param endDate the end date
     * @param type the type
     * @return the string
     */
    private String buildExportFile(String currentDate, String title) {
        StringBuilder builder = new StringBuilder();
        builder.append(currentDate);
        builder.append(title);
        return builder.toString();
    }

    /**
     * 设置导出文件响应头.
     *
     * @param fileName 文件名
     * @param response the response
     * @throws UnsupportedEncodingException the unsupported encoding exception
     */
    private void setExportResponse(String fileName, HttpServletResponse response) throws UnsupportedEncodingException {
        // 设置响应头和保存文件名
        response.setContentType("text/csv");
        response.setContentType("application/csv;charset=UTF-8");
        response.setHeader("Content-Disposition", "attachment; filename=" + URLEncoder.encode(fileName, "UTF-8")
                                                  + ".csv");
    }

    /**
     * 关闭输出流.
     *
     * @param out the out
     */
    private void closeOutputStream(OutputStream out) {
        try {
            if (out != null) {
                out.close();
            }
        } catch (IOException e) {
            logger.error("close OutputStream failed", e);
        }
    }

}
