/**
 * Project Name:media-deploy File Name:MediaController.java Package Name:cn.com.duiba.tuia.media.web.controller
 * Date:2016年9月26日上午11:03:32 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

import java.util.List;

import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import cn.com.duiba.tuia.media.api.dto.MediaAppDto;
import cn.com.duiba.tuia.media.api.dto.PageResultDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqCreatMediaAppDto;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqPageQueryMediaApp;
import cn.com.duiba.tuia.media.model.req.ReqUpdateMediaApp;
import cn.com.duiba.tuia.media.model.rsp.RspIdAndValue;
import cn.com.duiba.tuia.media.model.rsp.RspMediaApp;
import cn.com.duiba.tuia.media.service.MediaAppService;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.web.aop.LogWrite;

/**
 * ClassName:MediaController <br/>
 * Function: 媒体管理 Controller. <br/>
 * Date: 2016年9月26日 上午11:03:32 <br/>
 * 
 * @author guyan
 * @version
 * @since JDK 1.6
 * @see
 */
@Controller
@RequestMapping("/mediaApp/")
@Api("媒体相关api")
public class MediaAppController extends BaseController {

    /** The Media service. */
    @Autowired
    MediaAppService mediaAppService;

    /**
     * 查询媒体信息列表.
     *
     * @param param the param
     * @return the list
     */
    @ResponseBody
    @ApiOperation(value = "查询媒体信息列表", httpMethod = "GET", notes = "查询媒体信息列表接口")
    @RequestMapping(value = "getList", method = RequestMethod.GET)
    public Result<PageResultDto<RspMediaApp>> getList(@ModelAttribute ReqPageQueryMediaApp param) {
        try {
            param.setMediaId(RequestLocal.get().getCid());
            return successResult(mediaAppService.getMediaAppList(param));
        } catch (TuiaMediaException e) {
            logger.info("MediaAppController.getList error! the param=[{}]",param);
            return failResult(e);
        }
    }

    /**
     * 新增媒体信息.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "媒体模块", option = "创建媒体")
    @ResponseBody
    @ApiOperation(value = "新增媒体信息", httpMethod = "POST", notes = "新增媒体信息接口")
    @RequestMapping(value = "create", method = RequestMethod.POST)
    public Result<Boolean> create(@Valid @RequestBody ReqCreatMediaAppDto param, BindingResult result) {
        try {
            checkParam(result);
            param.setMediaId(RequestLocal.get().getCid());
            return successResult(this.mediaAppService.createMediaApp(param));
        } catch (TuiaMediaException e) {
            logger.error("MediaAppController.create error! the param=[{}]",param);
            return failResult(e);
        }
    }

    /**
     * 修改媒体信息.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */

    @LogWrite(modelName = "媒体模块", option = "修改媒体信息")
    @ResponseBody
    @ApiOperation(value = "修改媒体信息", httpMethod = "POST", notes = "修改媒体信息接口")
    @RequestMapping(value = "update", method = RequestMethod.POST)
    public Result<Boolean> update(@Valid @RequestBody ReqUpdateMediaApp param, BindingResult result) {
        try {
            checkParam(result);
            return successResult(this.mediaAppService.updateMediaApp(param));
        } catch (TuiaMediaException e) {
            logger.error("MediaAppController.update error! the param=[{}]",param);
            return failResult(e);

        }

    }

    /**
     * 获取媒体信息.
     *
     * @param appId the app id
     * @return the media app
     */
    @ResponseBody
    @ApiOperation(value = "获取媒体信息", httpMethod = "GET", notes = "获取媒体信息接口")
    @RequestMapping(value = "getMediaApp", method = RequestMethod.GET)
    public Result<MediaAppDto> getMediaApp(@RequestParam Long appId) {
        try {
            if (appId == null) {
                throw new TuiaMediaException(ErrorCode.E0401002);
            }
            return successResult(this.mediaAppService.getMediaAppDto(appId));
        } catch (TuiaMediaException e) {
            logger.error("MediaAppController.getMediaApp error! the appId=[{}]",appId);
            return failResult(e);
        }
    }

    @ResponseBody
    @ApiOperation(value = "获取所有的媒体用户appId和app名称", httpMethod = "GET", notes = "获取所有媒体用户appId和app名称信息接口")
    @RequestMapping(value = "getAllAppIdAndName", method = RequestMethod.GET)
    public Result<List<RspIdAndValue>> getAllAppIdAndName() {
        try {
            return successResult(this.mediaAppService.selectAppIdAndNameByMediaId(RequestLocal.get().getCid()));
        } catch (TuiaMediaException e) {
            logger.error("MediaAppController.getAllAppIdAndName error! the mediaId=[{}]",RequestLocal.get().getCid());
            return failResult(e);
        }

    }

}
