/**
 * Project Name:media-deploy<br/>
 * File Name:AbsenseController.java<br/>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br/>
 * Date:2016年9月26日下午5:02:40<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.web.controller;

import java.util.ArrayList;
import java.util.List;

import javax.validation.Valid;

import cn.com.duiba.tuia.media.api.dto.SlotDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqMaterialSpecificationDto;
import cn.com.duiba.tuia.media.api.dto.rsp.RspMaterialSpecificationDto;
import cn.com.duiba.tuia.media.service.MaterialSpecificationService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;

import cn.com.duiba.tuia.media.api.dto.SlotTypeParamDto;
import cn.com.duiba.tuia.media.api.dto.req.ReqUpdateSlot;
import cn.com.duiba.tuia.media.api.dto.rsp.RspSlotDetailDto;
import cn.com.duiba.tuia.media.bo.SlotBackendBo;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqCreateSlot;
import cn.com.duiba.tuia.media.model.req.ReqEditSlotEnableStatus;
import cn.com.duiba.tuia.media.model.req.ReqSlotList;
import cn.com.duiba.tuia.media.model.rsp.RspPageResult;
import cn.com.duiba.tuia.media.model.rsp.RspSlot;
import cn.com.duiba.tuia.media.service.SlotService;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.web.aop.LogWrite;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;

/**
 * 广告位相关controller <br/>
 * Date: 2016年9月26日 下午5:02:40 <br/>
 * .
 *
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */

@Controller
@RequestMapping("/slot")
@Api("广告位相关接口")
public class SlotController extends BaseController {

    /** The slot service. */
    @Autowired
    private SlotService                  slotService;

    @Autowired
    private SlotBackendBo                slotBackendBo;

    @Autowired
    private MaterialSpecificationService materialSpecificationService;

    /**
     * 创建广告位.
     *
     * @param req the req
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "创建广告位")
    @ResponseBody
    @ApiOperation(value = "创建广告位", httpMethod = "POST", notes = "创建广告位接口")
    @RequestMapping(value = "/create", method = RequestMethod.POST)
    public Result<Boolean> createSlot(@Valid @RequestBody ReqCreateSlot req, BindingResult result) {
        try {
            checkParam(result);
            RequestLocal rl = RequestLocal.get();
            req.setMediaId(rl.getCid());

            checkSlotType(req);

            return successResult(slotService.insert(req));
        } catch (Exception e) {
            logger.error("SlotController.createSlot error! the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * checkSlotType:(检查广告位类型与选择的尺寸). <br/>
     *
     * @author ZFZ
     * @param req
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private void checkSlotType(ReqCreateSlot req) throws TuiaMediaException {
        if (SlotDto.ADSENSE_TYPE_MATERIAL_SPEC == req.getSlotType()) {
            if (req.getMsId() == null) {
                throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "非手动投放类型规格不能为空！");
            }
        } else if (SlotDto.ADSENSE_TYPE_MANUAL == req.getSlotType()) {
            // Noop
        } else {
            throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "广告位类型未知");
        }
    }

    /**
     * Gets the list.
     *
     * @param param the param
     * @return the list
     */
    @ResponseBody
    @ApiOperation(value = "查询广告位列表", httpMethod = "GET", notes = "查询广告位列表")
    @RequestMapping(value = "/getList", method = RequestMethod.GET)
    public Result<RspPageResult<RspSlot>> getList(@ModelAttribute ReqSlotList param) {
        try {
            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            param.setPageSize(ReqSlotList.PAGE_SIZE_50);
            param.setRowStart((param.getCurrentPage() - 1) * param.getPageSize());
            return successResult(slotService.selectListByPage(param));
        } catch (Exception e) {
            logger.error("SlotController.getList error! the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 查看广告位详情.
     *
     * @param slotId the slot id
     * @return the detail
     */
    @ApiOperation(value = "查看广告位详情", httpMethod = "GET", notes = "查看广告位详情接口")
    @ApiImplicitParam(name = "slotId", value = "广告位ID", required = true, dataType = "Long", paramType = "query")
    @ResponseBody
    @RequestMapping("/getDetail")
    public Result<RspSlotDetailDto> getDetail(@RequestParam Long slotId) {
        try {
            if (slotId == null) {
                logger.error("slotIds can not be null");
                throw new TuiaMediaException(ErrorCode.E0001005);
            }
            return successResult(slotBackendBo.getSlotDetailById(slotId));
        } catch (Exception e) {
            logger.error("slotController getDetail error! the slotId=[{}]", slotId);
            return failResult(e);
        }
    }

    /**
     * 编辑广告位.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "编辑广告位")
    @ResponseBody
    @ApiOperation(value = "编辑广告位", httpMethod = "POST", notes = "编辑广告位")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result<Boolean> update(@Valid @RequestBody ReqUpdateSlot param, BindingResult result) {
        try {
            checkParam(result);

            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            return successResult(slotService.update(param));
        } catch (Exception e) {
            logger.error("SlotController.update error! the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 广告位开启/关闭.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "广告位模块", option = "广告位开启/关闭")
    @ResponseBody
    @ApiOperation(value = "广告位开启/关闭", httpMethod = "POST", notes = "广告位开启/关闭")
    @RequestMapping(value = "/updateEnableStatus", method = RequestMethod.POST)
    public Result<Boolean> updateEnableStatus(@Valid @RequestBody ReqEditSlotEnableStatus param, BindingResult result) {
        try {
            checkParam(result);
            RequestLocal rl = RequestLocal.get();
            param.setMediaId(rl.getCid());
            return successResult(slotService.updateEnableStatus(param));
        } catch (Exception e) {
            logger.error("SlotController.updateEnableStatus error! the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * getSlotTypeParam:查询广告类型参数. <br/>
     *
     * @author guyan
     * @return
     * @since JDK 1.6
     */
    @ResponseBody
    @ApiOperation(value = "查询广告类型参数", httpMethod = "GET", notes = "查询广告类型参数")
    @RequestMapping(value = "/getSlotTypeParam", method = RequestMethod.GET)
    public Result<List<SlotTypeParamDto>> getSlotTypeParam() {
        try {
            List<SlotTypeParamDto> result = new ArrayList<>();
            SlotTypeParamDto manual = new SlotTypeParamDto();
            manual.setTypeCode(SlotDto.ADSENSE_TYPE_MANUAL);
            manual.setSlotType(SlotDto.ADSENSE_TYPE_MANUAL_STR);
            result.add(manual);
            SlotTypeParamDto notManual = new SlotTypeParamDto();
            notManual.setTypeCode(SlotDto.ADSENSE_TYPE_MATERIAL_SPEC);
            notManual.setSlotType(SlotDto.ADSENSE_TYPE_MATERIAL_SPEC_STR);
            result.add(notManual);
            return successResult(result);
        } catch (Exception e) {
            logger.error("SlotController.getSlotTypeParam error!");
            return failResult(e);
        }
    }

    /**
     * 查询规格列表
     * 
     * @return 规格列表
     */
    @ResponseBody
    @ApiOperation(value = "查询规格列表", httpMethod = "GET", notes = "查询规格列表")
    @RequestMapping(value = "/getMsList", method = RequestMethod.GET)
    public Result<List<RspMaterialSpecificationDto>> getMsList() {
        try {
            ReqMaterialSpecificationDto req = new ReqMaterialSpecificationDto();
            req.setSort("gmt_modified");
            req.setOrder("desc");
            req.setPageSize(-1);
            List<RspMaterialSpecificationDto> list = materialSpecificationService.getMaterialSpecificationList(req);
            return successResult(list);
        } catch (Exception e) {
            logger.error("SlotController.getMsList error!");
            return failResult(e);
        }

    }
}
