/**
 * Project Name:media-deploy<br/>
 * File Name:LoginInterceptor.java<br/>
 * Package Name:cn.com.duiba.tuia.media.web.filter<br/>
 * Date:2016年12月9日下午3:52:19<br/>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.media.web.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import cn.com.duiba.tuia.media.api.dto.AccountDto;
import cn.com.duiba.tuia.media.common.constants.CommonConstants.CookieKey;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.tool.RequestTool;
import cn.com.duiba.tuia.media.common.tool.TimeProfileTool;
import cn.com.duiba.tuia.media.common.utils.AjaxUtils;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.utils.CookieUtil;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.utils.ResultUtil;
import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * ClassName:LoginInterceptor <br/>
 * Date: 2016年12月9日 下午3:52:19 <br/>
 * 
 * @author ZFZ
 * @version
 * @since JDK 1.6
 * @see
 */
@Component
public class LoginInterceptor  extends  HandlerInterceptorAdapter implements InitializingBean {

    /** The logger. */
    protected Logger            logger           = LoggerFactory.getLogger(getClass());

    /** 权限过了列表(不需要做权限校验). */
    private static List<String> notCheckUrlList = new ArrayList<>();

    @Autowired
    private AccountService      accountService;
    
    
    
    @Override
    public void afterPropertiesSet() throws Exception {
        //
    }
    

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        TimeProfileTool.enter("LoginInterceptor.preHandle");

        RequestTool.setRequestInThreadLocal(request);

        RequestLocal.clear();
        RequestLocal.get().setRequest(request);
        RequestLocal.get().setResponse(response);

        TimeProfileTool.release();
        
        return doCheckPath(request, response);
    }

    private boolean doCheckPath(HttpServletRequest request, HttpServletResponse response) throws IOException {
        Long cid = RequestLocal.get().getCid();
        String path = request.getServletPath();

        if (path.startsWith("/webjars") || path.startsWith("/swagger") || notCheckUrlList.contains(path)) {
            // 免拦截页面或请求
        } else if (cid != null) {

            if ("/account/login".equals(path) || "/account/register".equals(path)) {
                // 登录过且未失效, 再重新登录的话重定向到首页
                logger.error("please enter the home page! the url is:" + path);
                response.sendRedirect("/#/private");
                return false;
            }

            // 校验用户状态
            try {
                accountStatusAuth(cid);
            } catch (TuiaMediaException e) {
                // 清除cookie
                CookieUtil.deleteCookie(CookieKey.ACCOUNT_KEY);
                logger.error("please login again! the url is:" + path);
                exceptionFailure(response, e);
                return false;
            }

        } else if ("/private".equals(path)) {
            response.sendRedirect("/#/signin");
            return false;
        } else if (!notCheckUrlList.contains(path)) {
            logger.error("please login again! the url is:" + path);
            AjaxUtils.renderJson(response, ResultUtil.fail(ErrorCode.E9999998));
            return false;
        }

        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
                                                                                                                       throws Exception {
        TimeProfileTool.release();
    }

    /**
     * 校验账号状态.
     *
     * @param accountId the account id
     * @param auditStatus 账号审核状态
     * @param freezeStatus 冻结状态
     * @throws TuiaException 失败抛出异常
     */
    private void accountStatusAuth(long mediaId) throws TuiaMediaException {
        AccountDto accountDto = accountService.selectByIdNotNull(mediaId);
        doCheckAudit(accountDto.getCheckStatus());
        doCheckFreeze(accountDto.getFreezeStatus());
    }

    /**
     * 校验账户是否冻结. <br/>
     *
     * @author xiawei
     * @param freezeStatus
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private void doCheckFreeze(Integer freezeStatus) throws TuiaMediaException {
        if (AccountDto.FREEZED_STATUS == freezeStatus) {
            // 被冻结状态
            logger.error("the account is freeze, the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102004);
        }
        if (AccountDto.UNFREEZED_STATUS != freezeStatus) {
            // 非法的冻结状态(既不是冻结也不是未冻结)
            logger.error("the account freeze status is illegal,  the freeze status=[{}]", freezeStatus);
            throw new TuiaMediaException(ErrorCode.E0102005);
        }
    }

    /**
     * 校验账户是否审核. <br/>
     *
     * @author xiawei
     * @param auditStatus
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    private void doCheckAudit(Integer auditStatus) throws TuiaMediaException {
        if (AccountDto.CHECK_STATUS_ING == auditStatus) {
            // 审核中状态
            logger.error("user account in review status, audit status=[{}]", auditStatus,
                         auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102006);
        }
        if (AccountDto.CHECK_STATUS_REFUSE == auditStatus) {
            // 被拒绝状态
            logger.error("user account in refuse status, and audit status=[{}]", auditStatus,
                         auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102007);
        }
        if (AccountDto.CHECK_STATUS_PASS != auditStatus) {
            // 非法的审核状态
            logger.error("the account status is illegal, and audit status=[{}]", auditStatus,
                         auditStatus);
            throw new TuiaMediaException(ErrorCode.E0102008);
        }
    }

    /**
     * exceptionFailure:(构造失败结果返回). <br/>
     *
     * @author ZFZ
     * @param response
     * @param e
     * @since JDK 1.6
     */
    public void exceptionFailure(HttpServletResponse response, Exception e) {
        if (e instanceof TuiaMediaException) {
            AjaxUtils.renderJson(response,
                                 ResultUtil.fail(((TuiaMediaException) e).getResultCode(),
                                                 ((TuiaMediaException) e).getResultMessage()));
        } else {
            logger.error("系统错误", e);
            AjaxUtils.renderJson(response, ResultUtil.fail(ErrorCode.E9999999));
        }
    }

    protected <T> void doTuiaMediaException(DubboResult<T> result, String callInterface) throws TuiaMediaException {
        if (!result.isResultSuccess()) {
            String returnCode = result.getReturnCode();
            String msg = result.getMsg();
            // 创建失败
            logger.error("[TuiaMedia] " + callInterface + " failed, because of=[{}] and the returnCode=[{}]",
                         result.getMsg(), returnCode);
            if (returnCode.endsWith("TM-9999999")) {
                // 如果不是tuia-core发生的未知错误， 则把错误描述返回给前端
                msg = "发生未知错误";
                returnCode = "9999999";
            }
            throw new TuiaMediaException(returnCode, msg);
        }
    }

    /**
     * Inits the.
     *
     * @param fc the fc
     * @throws ServletException the servlet exception
     */
    static {
        // 前台路径
        notCheckUrlList.add("/");
        notCheckUrlList.add("/private");
        notCheckUrlList.add("/index.html");
        notCheckUrlList.add("/favicon.ico");
        notCheckUrlList.add("/#/signin");
        notCheckUrlList.add("/private#/404");
        notCheckUrlList.add("/__webpack_hmr");
        // swagger
        notCheckUrlList.add("/swagger-ui.html");
        notCheckUrlList.add("/images/favicon-16x16.png");
        notCheckUrlList.add("/swagger-resources");
        notCheckUrlList.add("/v2/api-docs");
        notCheckUrlList.add("/images/favicon-32x32.png");
        notCheckUrlList.add("/images/favicon-16x16.png");
        notCheckUrlList.add("/configuration/security");
        notCheckUrlList.add("/configuration/ui");

        // 上传文件接口
        notCheckUrlList.add("/upload/index");

        // 用户相关接口
        notCheckUrlList.add("/account/resetPasswd/redirect");
        notCheckUrlList.add("/account/sendResetPasswdEmail");
        notCheckUrlList.add("/account/resetPasswd");
        notCheckUrlList.add("/account/isExists");
        notCheckUrlList.add("/account/isExistPhone");
        notCheckUrlList.add("/account/register");
        notCheckUrlList.add("/account/login");
        notCheckUrlList.add("/account/verifyEmail");
        notCheckUrlList.add("/account/repeatVerifyEmail");
        notCheckUrlList.add("/account/updateAuditData");
        notCheckUrlList.add("/account/getUncheckAccount");
        notCheckUrlList.add("/account/sendResetPdEmail");
        notCheckUrlList.add("/account/verifyResetPdCode");
        notCheckUrlList.add("/account/resetPassword");
        notCheckUrlList.add("/account/logout");

        // 短信相关接口
        notCheckUrlList.add("/sms/send");
        notCheckUrlList.add("/sms/verify");
        
        // 测试相关接口
        notCheckUrlList.add("/remoteTest/getListDetail");
        notCheckUrlList.add("/remoteTest/updateBatchSlotCache");
        notCheckUrlList.add("/remoteTest/updateStrategy");
        notCheckUrlList.add("/remoteTest/getSlot");
        notCheckUrlList.add("/remoteTest/updateSlot");
        notCheckUrlList.add("/remoteTest/getStrategy");
        notCheckUrlList.add("/remoteTest/isValidMediaApp");
        notCheckUrlList.add("/remoteTest/getMediaApp");
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
                           ModelAndView modelAndView) throws Exception {
        
        //
    }
}