/**
 * Project Name:media-deploy<br>
 * File Name:AccountController.java<br>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br>
 * Date:2016年9月26日上午10:38:36<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */
package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.util.Date;

import javax.validation.Valid;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import cn.com.duiba.tuia.media.bo.AccountBackendBO;
import cn.com.duiba.tuia.media.common.constants.CommonConstants.CookieKey;
import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;
import cn.com.duiba.tuia.media.common.utils.BlowfishUtils;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqEmailToken;
import cn.com.duiba.tuia.media.model.req.ReqLogin;
import cn.com.duiba.tuia.media.model.req.ReqRegister;
import cn.com.duiba.tuia.media.model.req.ReqResetPassword;
import cn.com.duiba.tuia.media.model.req.ReqSendEmail;
import cn.com.duiba.tuia.media.model.req.ReqSendSMS;
import cn.com.duiba.tuia.media.model.req.ReqUpdateAccountBase;
import cn.com.duiba.tuia.media.model.req.ReqUpdateCompanyInfo;
import cn.com.duiba.tuia.media.model.req.ReqUpdatePersonInfo;
import cn.com.duiba.tuia.media.model.req.ReqUpdateVerifyInfo;
import cn.com.duiba.tuia.media.model.req.ReqVerifyCode;
import cn.com.duiba.tuia.media.model.rsp.AccountInfoRsp;
import cn.com.duiba.tuia.media.model.rsp.RegisterRsp;
import cn.com.duiba.tuia.media.model.rsp.TokenRsp;
import cn.com.duiba.tuia.media.service.AccountService;
import cn.com.duiba.tuia.media.service.SmsService;
import cn.com.duiba.tuia.media.utils.CookieUtil;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.web.aop.LogWrite;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName: AccountController <br/>
 * date: 2016年9月26日 上午10:38:36 <br/>
 * .
 *
 * @author xiawei
 * @version
 * @since JDK 1.6
 */
@Controller
@RequestMapping("/account")
@Api("账户相关api")
public class AccountController extends BaseController {

    /** 账号信息部分的加密秘钥. */
    @Value("${media.consumer.encrypt.key}")
    private String           consumerEncryptKey;

    /** The account service. */
    @Autowired
    private AccountService   accountService;

    /** The account backend bo. */
    @Autowired
    private AccountBackendBO accountBackendBO;

    @Autowired
    private SmsService       smsService;

    /** 更换手机号. */
    private static final int REPLACE_PHONE_NUM = 2;

    /**
     * isEmailExists:(媒体账号是否被占用). <br/>
     *
     * @author ZFZ
     * @param value
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "媒体账号是否被占用", httpMethod = "GET", notes = "媒体账号是否被占用接口")
    @ResponseBody
    @RequestMapping(value = "/isExists", method = RequestMethod.GET)
    public Result<Boolean> isEmailExists(@RequestParam String value) {
        try {
            // 参数校验
            if (accountService.isEmailExists(value)) {
                logger.error("email is exist, email =[{}]", value);
                throw new TuiaMediaException(ErrorCode.E0102011);
            }
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.isEmialExists error!,the param=[{}]", value);
            return failResult(e);
        }
    }

    /**
     * isExistPhone:(注册时手机号是否被占用). <br/>
     *
     * @author ZFZ
     * @param value
     * @param mediaId
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "注册时手机号是否被占用", httpMethod = "GET", notes = "注册时手机号是否被占用接口")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "value", value = "手机号", required = true, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "mediaId", value = "媒体账户ID", required = false, dataType = "Long", paramType = "query") })
    @ResponseBody
    @RequestMapping(value = "/isExistPhone", method = RequestMethod.GET)
    public Result<Boolean> isExistPhone(@RequestParam String value, @RequestParam(required = false) Long mediaId) {
        try {
            if (accountService.repeatRegisterVerifyPhone(value, null)) {
                logger.error("linkPhone is exist, linkPhone =[{}]", value);
                throw new TuiaMediaException(ErrorCode.E0102021);
            }
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.isExistPhone error!,the param=[{}]", value);
            return failResult(e);
        }
    }

    /**
     * 账号注册.
     *
     * @param req the req
     * @param result the result
     * @return the result< register rsp>
     */
    @LogWrite(modelName = "用户模块", option = "注册", ignoreParams = { "passwd" })
    @ApiOperation(value = "媒体账号注册", httpMethod = "POST", notes = "媒体账号注册接口")
    @ResponseBody
    @RequestMapping(value = "/register", method = RequestMethod.POST)
    public Result<RegisterRsp> register(@Valid @RequestBody ReqRegister req, BindingResult result) {
        try {
            // 校验参数
            checkParam(result);
            return successResult(accountBackendBO.register(req));
        } catch (Exception e) {
            logger.error("AccountController.register error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * verifyEmail:(注册邮件验证). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "注册邮件验证", httpMethod = "GET", notes = "注册邮件验证接口")
    @RequestMapping(value = "/verifyEmail", method = RequestMethod.GET)
    public String verifyEmail(@Valid @ModelAttribute ReqEmailToken req, BindingResult result) {
        try {
            // 校验参数
            checkParam(result);
            accountService.verifyEmail(req.getToken());
            return "redirect:/#/verified?email=" + req.getEmail();
        } catch (Exception e) {
            logger.error("verifyEmail error!,the param = [{}] and the msg = [{}]", req, e);
            return "redirect:/#/expired?email=" + req.getEmail();

        }
    }

    /**
     * repeatVerifyEmail:(重新发送验证邮件.). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "重新发送验证邮件", httpMethod = "POST", notes = "重新发送验证邮件接口")
    @ResponseBody
    @RequestMapping(value = "/repeatVerifyEmail", method = RequestMethod.POST)
    public Result<Boolean> repeatVerifyEmail(@Valid @RequestBody ReqSendEmail req, BindingResult result) {
        try {
            // 校验参数
            checkParam(result);
            accountService.repeatVerifyEmail(req.getEmail());
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.repeatVerifyEmail error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * login:(登录.). <br/>
     *
     * @author ZFZ
     * @param reqLogin
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "用户模块", option = "登录", ignoreParams = { "password" })
    @ApiOperation(value = "媒体账号登录", httpMethod = "POST", notes = "媒体账号登录接口")
    @ResponseBody
    @RequestMapping(value = "/login", method = RequestMethod.POST)
    public Result<Boolean> login(@Valid @RequestBody ReqLogin reqLogin, BindingResult result) {
        try {
            // 校验参数
            checkParam(result);

            // 构建cookie
            buildLoginCookie(accountBackendBO.login(reqLogin.getEmail(), reqLogin.getPassword()));

            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.login error!,the param=[{}]", reqLogin);
            return failResult(e);
        }
    }

    /**
     * sendResetPdEmail:(发送重置密码验证码邮件). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "发送重置密码验证码邮件", httpMethod = "POST", notes = "发送重置密码验证码邮件接口")
    @ResponseBody
    @RequestMapping(value = "/sendResetPdEmail", method = RequestMethod.POST)
    public Result<Boolean> sendResetPdEmail(@Valid @RequestBody ReqSendEmail req, BindingResult result) {
        try {
            // 校验参数
            checkParam(result);
            return successResult(accountService.sendResetPasswdEmail(req.getEmail()));
        } catch (Exception e) {
            logger.error("AccountController.sendResetPdEmail error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * 发送重置密码验证码邮件.<br>
     *
     * @param req the req
     * @param result the result
     * @return the result< boolean>
     */
    @ApiOperation(value = "发送更换邮箱验证码邮件", httpMethod = "POST", notes = "发送更换邮箱验证码邮件接口")
    @ResponseBody
    @RequestMapping(value = "/sendReplaceEmail", method = RequestMethod.POST)
    public Result<Boolean> sendReplaceEmail(@Valid @RequestBody ReqSendEmail req, BindingResult result) {
        try {
            checkParam(result);
            return successResult(accountService.sendReplaceEmail(req.getEmail(), RequestLocal.get().getCid()));
        } catch (Exception e) {
            logger.error("AccountController.sendReplaceEmail error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * verifyResetPdCode:(重置密码 验证码校验). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @ApiOperation(value = "重置密码验证码校验", httpMethod = "POST", notes = "重置密码验证码校验接口")
    @ResponseBody
    @RequestMapping(value = "/verifyResetPdCode", method = RequestMethod.POST)
    public Result<TokenRsp> verifyResetPdCode(@Valid @RequestBody ReqVerifyCode req, BindingResult result) {
        try {
            checkParam(result);

            return successResult(accountService.resetPasswdRedirect(req.getEmail(), req.getVerificationCode()));

        } catch (Exception e) {
            logger.error("AccountController.verifyResetPdCode error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * resetPassword:(重置密码). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "用户模块", option = "重置密码", ignoreParams = { "newPassword" })
    @ApiOperation(value = "重置密码", httpMethod = "POST", notes = "重置密码接口")
    @ResponseBody
    @RequestMapping(value = "/resetPassword", method = RequestMethod.POST)
    public Result<Boolean> resetPassword(@Valid @RequestBody ReqResetPassword req, BindingResult result) {
        try {
            checkParam(result);
            boolean flag = accountService.resetPassword(req.getEmail(), req.getNewPassword(), req.getToken());
            // 重置密码成功后删除cookie
            CookieUtil.deleteCookie(CookieKey.ACCOUNT_KEY);
            return successResult(flag);
        } catch (Exception e) {
            logger.error("AccountController.resetPasswd error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * 查询用户详情.
     *
     * @return the account info
     */
    @ApiOperation(value = "查询用户详情", httpMethod = "GET", notes = "查询用户详情接口")
    @ResponseBody
    @RequestMapping(value = "/getAccountInfo", method = RequestMethod.GET)
    public Result<AccountInfoRsp> getAccountInfo() {
        try {
            Long mediaId = RequestLocal.get().getCid();
            return successResult(accountBackendBO.getAccountInfo(mediaId));
        } catch (Exception e) {
            logger.error("AccountController.getAccountInfo error!");
            return failResult(e);
        }
    }

    /**
     * updateAccountBase:(修改用户基本信息). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "用户模块", option = "修改用户基本信息")
    @ApiOperation(value = "修改用户基本信息", httpMethod = "POST", notes = "修改用户基本信息接口")
    @ResponseBody
    @RequestMapping(value = "/updateAccountBase", method = RequestMethod.POST)
    public Result<Boolean> updateAccountBase(@Valid @RequestBody ReqUpdateAccountBase req, BindingResult result) {
        try {
            checkParam(result);
            req.setMediaId(RequestLocal.get().getCid());
            return successResult(accountService.updateAccountBase(req));
        } catch (Exception e) {
            logger.error("AccountController.updateAccountBase error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * 媒体账户退出.
     *
     * @return the result< boolean>
     */
    @ApiOperation(value = "媒体账户退出", httpMethod = "POST", notes = "媒体账户退出接口")
    @ResponseBody
    @RequestMapping(value = "/logout", method = RequestMethod.POST)
    public Result<Boolean> logout() {
        try {
            CookieUtil.deleteCookie(CookieKey.ACCOUNT_KEY);
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.logout error!");
            return failResult(e);
        }
    }

    /**
     * 构建登录成功后的cookie信息.<br>
     * <ol>
     * <li>loginTime:登录成功的时间</li>
     * <li>accountId:账号ID</li>
     * </ol>
     *
     * @param mediaId the media id
     */
    private void buildLoginCookie(Long mediaId) {
        JSONObject jsonObject = new JSONObject();
        // 登陆时间
        jsonObject.put(CookieKey.LOGIN_TIME_KEY, new Date().getTime());
        // 账号ID
        jsonObject.put(CookieKey.ACCOUNT_ID_KEY, mediaId);

        String envalue = BlowfishUtils.encryptBlowfish(jsonObject.toJSONString(), consumerEncryptKey);
        CookieUtil.setCrossCookie(CookieKey.ACCOUNT_KEY, envalue);
    }

    /**
     * updateCompanyInfo:(用户修改公司信息). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "用户模块", option = "用户修改公司信息")
    @ApiOperation(value = "用户修改公司信息", httpMethod = "POST", notes = "用户修改公司信息接口")
    @ResponseBody
    @RequestMapping(value = "/updateCompanyInfo", method = RequestMethod.POST)
    public Result<Boolean> updateCompanyInfo(@Valid @RequestBody ReqUpdateCompanyInfo req, BindingResult result) {
        try {
            checkParam(result);
            req.setMediaId(RequestLocal.get().getCid());
            accountService.updateCompanyInfo(req);
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.updateCompanyInfo error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * updatePersonInfo:(用户修改个人信息). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "用户模块", option = "用户修改个人信息")
    @ApiOperation(value = "用户修改个人信息", httpMethod = "POST", notes = "用户修改个人信息接口")
    @ResponseBody
    @RequestMapping(value = "/updatePersonInfo", method = RequestMethod.POST)
    public Result<Boolean> updatePersonInfo(@Valid @RequestBody ReqUpdatePersonInfo req, BindingResult result) {
        try {
            checkParam(result);
            req.setMediaId(RequestLocal.get().getCid());
            accountService.updatePersonInfo(req);
            return successResult(true);
        } catch (Exception e) {
            logger.error("AccountController.updatePersonInfo error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    @ApiOperation(value = "发送更换手机号短信验证码", httpMethod = "POST", notes = "发送更换手机号短信验证码接口")
    @ResponseBody
    @RequestMapping(value = "/sendReplacePhone", method = RequestMethod.POST)
    public Result<Boolean> sendReplacePhone(@Valid @RequestBody ReqSendSMS req, BindingResult result) {
        try {
            checkParam(result);
            req.setMediaId(RequestLocal.get().getCid());
            return successResult(accountService.sendReplacePhone(req));
        } catch (Exception e) {
            logger.error("AccountController.sendReplaceEmail error!,the param=[{}]", req);
            return failResult(e);
        }

    }

    @ApiOperation(value = "修改手机号", httpMethod = "POST", notes = "修改手机号接口")
    @ResponseBody
    @RequestMapping(value = "/updatePhoneNum", method = RequestMethod.POST)
    public Result<Boolean> updatePhoneNum(@Valid @RequestBody ReqUpdateVerifyInfo req, BindingResult result) {
        try {
            checkParam(result);
            Long mediaId = RequestLocal.get().getCid();
            String linkPhone = req.getLinkPhone();
            if (StringUtils.isEmpty(linkPhone)) {
                logger.info("linkPhone is not null,the linkPhone=[{}]]", linkPhone);
                throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "电话号不能为空！");
            }
            if (accountService.repeatRegisterVerifyPhone(linkPhone, mediaId)) { //判断手机号是否已经注册
                logger.error("linkPhone is exist, linkPhone =[{}]", linkPhone);
                throw new TuiaMediaException(ErrorCode.E0102021);
            }
            smsService.verify(REPLACE_PHONE_NUM, linkPhone, req.getCode()); //验证短信验证码
            return successResult(accountService.updatePhoneNum(mediaId, linkPhone));
        } catch (Exception e) {
            logger.error("AccountController.sendReplaceEmail error!,the req=[{}]", req);
            return failResult(e);
        }
    }

    @ApiOperation(value = "修改邮箱", httpMethod = "POST", notes = "修改邮箱接口")
    @ResponseBody
    @RequestMapping(value = "/updateEmail", method = RequestMethod.POST)
    public Result<Boolean> updateEmail(@Valid @RequestBody ReqUpdateVerifyInfo req, BindingResult result) {
        try {
            checkParam(result);
            String email = req.getEmail();
            if (StringUtils.isEmpty(email)) {
                logger.info("email is not null,the email=[{}]]", email);
                throw new TuiaMediaException(ErrorCode.E0001005.getErrorCode(), "邮箱地址不能为空！");
            }
            if (accountService.isEmailExists(email)) {  //判断邮箱是否已存在
                logger.error("email is exist, the email =[{}]", email);
                throw new TuiaMediaException(ErrorCode.E0102011);
            }
            return successResult(accountService.replaceEmail(req.getCode(), RequestLocal.get().getCid()));
        } catch (Exception e) {
            logger.error("AccountController.sendReplaceEmail error!,the req=[{}]", req);
            return failResult(e);
        }
    }

}
