/**
 * Project Name:media-deploy<br>
 * File Name:StrategyController.java<br>
 * Package Name:cn.com.duiba.tuia.media.web.controller<br>
 * Date:2016年9月27日上午11:22:05<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.media.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import cn.com.duiba.tuia.media.api.dto.BaseQueryDto;
import cn.com.duiba.tuia.media.model.Result;
import cn.com.duiba.tuia.media.model.req.ReqDelete;
import cn.com.duiba.tuia.media.model.req.ReqShieldListStrategy;
import cn.com.duiba.tuia.media.model.req.ReqStrategyPageQuery;
import cn.com.duiba.tuia.media.model.rsp.RspIdAndValue;
import cn.com.duiba.tuia.media.model.rsp.RspPageResult;
import cn.com.duiba.tuia.media.model.rsp.RspShieldStrategy;
import cn.com.duiba.tuia.media.model.rsp.RspStrategyList;
import cn.com.duiba.tuia.media.service.StrategyService;
import cn.com.duiba.tuia.media.utils.RequestLocal;
import cn.com.duiba.tuia.media.web.aop.LogWrite;

/**
 * ClassName: StrategyController <br/>
 * Function: 屏蔽策略Controller. <br/>
 * date: 2016年9月27日 上午11:22:05 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Controller
@RequestMapping("/strategy")
@Api("屏蔽策略相关api")
public class StrategyController extends BaseController {

    /** The strategy service. */
    @Autowired
    private StrategyService strategyService;

    /**
     * 创建屏蔽策略.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "屏蔽策略模块", option = "创建屏蔽策略")
    @ResponseBody
    @ApiOperation(value = "创建屏蔽策略", httpMethod = "POST", notes = "创建屏蔽策略接口")
    @RequestMapping(value = "/create", method = RequestMethod.POST)
    public Result<Boolean> create(@Valid @RequestBody ReqShieldListStrategy param, BindingResult result) {
        try {
            // 参数校验
            checkParam(result);
            return successResult(strategyService.create(param, RequestLocal.get().getCid()));
        } catch (Exception e) {
            logger.error("StrategyController.create error!,the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 更新屏蔽策略.
     *
     * @param param the param
     * @param result the result
     * @return the result< boolean>
     */
    @LogWrite(modelName = "屏蔽策略模块", option = "更新屏蔽策略")
    @ResponseBody
    @ApiOperation(value = "更新屏蔽策略", httpMethod = "POST", notes = "更新屏蔽策略接口")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result<Boolean> update(@Valid @RequestBody ReqShieldListStrategy param, BindingResult result) {
        try {
            // 参数校验
            checkParam(result);
            return successResult(strategyService.update(param, RequestLocal.get().getCid()));
        } catch (Exception e) {
            logger.error("StrategyController.update error!,the param=[{}]", param);
            return failResult(e);
        }
    }

    /**
     * 
     * delete:(删除屏蔽策略.). <br/>
     *
     * @author ZFZ
     * @param req
     * @param result
     * @return
     * @since JDK 1.6
     */
    @LogWrite(modelName = "屏蔽策略模块", option = "删除屏蔽策略")
    @ResponseBody
    @ApiOperation(value = "删除屏蔽策略", httpMethod = "POST", notes = "删除屏蔽策略接口")
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public Result<Boolean> delete(@RequestBody ReqDelete req, BindingResult result) {
        try {
            // 参数校验
            checkParam(result);
            return successResult(strategyService.delete(RequestLocal.get().getCid(), req.getId()));
        } catch (Exception e) {
            logger.error("StrategyController.delete error!,the param=[{}]", req);
            return failResult(e);
        }
    }

    /**
     * 查询屏蔽策略详情.
     *
     * @param id the id
     * @return the detail
     */
    @ResponseBody
    @ApiOperation(value = "查询屏蔽策略详情", httpMethod = "GET", notes = "查询屏蔽策略详情接口")
    @RequestMapping(value = "/getDetail", method = RequestMethod.GET)
    public Result<RspShieldStrategy> getDetail(@RequestParam Long id) {
        try {
            return successResult(strategyService.getDetail(id));
        } catch (Exception e) {
            logger.error("StrategyController.getDetail error!,the param=[{}]", id);
            return failResult(e);
        }
    }

    /**
     * 查询屏蔽策略列表.
     *
     * @return the list by media id
     */
    @ResponseBody
    @ApiOperation(value = "查询屏蔽策略列表", httpMethod = "GET", notes = "查询屏蔽策略列表接口")
    @RequestMapping(value = "/getListByMediaId", method = RequestMethod.GET)
    public Result<List<RspIdAndValue>> getListByMediaId() {
        try {
            return successResult(strategyService.getListByMediaId(RequestLocal.get().getCid()));
        } catch (Exception e) {
            logger.error("StrategyController.getList error!");
            return failResult(e);
        }
    }

    /**
     * Gets the page list.
     *
     * @param req the req
     * @return the page list
     */
    @ResponseBody
    @ApiOperation(value = "分页查询屏蔽策略", httpMethod = "GET", notes = "分页查询屏蔽策略接口")
    @RequestMapping(value = "/getPageList", method = RequestMethod.GET)
    public Result<RspPageResult<RspStrategyList>> getPageList(@ModelAttribute ReqStrategyPageQuery req) {
        try {
            if (req.getPageSize() == null) {
                req.setPageSize(BaseQueryDto.PAGE_SIZE_50);
            }

            req.setRowStart((req.getCurrentPage() - 1) * req.getPageSize());
            req.setMediaId(RequestLocal.get().getCid());
            return successResult(strategyService.getPageList(req));
        } catch (Exception e) {
            logger.error("StrategyController.getPageList error!,the param=[{}]", req);
            return failResult(e);
        }
    }
}
