package cn.com.duiba.tuia.purchase.web.api.localservice;

import cn.com.duiba.tuia.purchase.web.api.constant.MediaCallbackStrategyConsts;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpEventCallbackStrategyContext;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpMediaCallbackStrategyDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpMediaPlatformDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpProductDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpTaskDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PutPlanDto;
import cn.com.duiba.tuia.purchase.web.api.remoteservice.RemotePurchaseService;
import com.alibaba.fastjson.JSON;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class OcpcPurchaseService {

    @Resource
    private RemotePurchaseService remotePurchaseService;

    private final LoadingCache<Long, Optional<PutPlanDto>> planCaffeineCache = Caffeine.newBuilder()
            .maximumSize(4000)
            .initialCapacity(20)
            .refreshAfterWrite(30, TimeUnit.SECONDS)
            .expireAfterWrite(5 * 60, TimeUnit.SECONDS)
            .build(this::queryByChannelId);

    private final LoadingCache<Long, Optional<PbpTaskDto>> taskCaffeineCache = Caffeine.newBuilder()
            .maximumSize(4000)
            .initialCapacity(20)
            .refreshAfterWrite(30, TimeUnit.SECONDS)
            .expireAfterWrite(5 * 60, TimeUnit.SECONDS)
            .build(this::queryByTaskId);

    private final LoadingCache<Long, Optional<PbpProductDto>> productCaffeineCache = Caffeine.newBuilder()
            .maximumSize(1000)
            .initialCapacity(10)
            .refreshAfterWrite(30, TimeUnit.SECONDS)
            .expireAfterWrite(5 * 60, TimeUnit.SECONDS)
            .build(this::queryByProductId);

    private final LoadingCache<Long, Optional<PbpMediaPlatformDto>> mediaPlatformCaffeineCache = Caffeine.newBuilder()
            .maximumSize(200)
            .initialCapacity(10)
            .refreshAfterWrite(30 * 60, TimeUnit.SECONDS)
            .expireAfterWrite(2 * 60 * 60, TimeUnit.SECONDS)
            .build(this::queryByMediaPlatformId);

    /**
     * 根据渠道ID查询投放计划
     * @param channelId 渠道ID（实际是任务ID）
     * @return 投放计划
     */
    public Optional<PutPlanDto> getByChannelId(Long channelId) {
        return planCaffeineCache.get(channelId);
    }

    /**
     * 根据任务ID查询产品信息
     * @param taskId 任务ID
     * @return 产品信息
     */
    public Optional<PbpProductDto> getProductByTaskId(Long taskId) {
        Optional<PbpTaskDto> pbpTaskDto = getTaskById(taskId);
        if (!pbpTaskDto.isPresent()) {
            return Optional.empty();
        }
        Long productId = pbpTaskDto.get().getProductId();
        return getByProductId(productId);
    }

    /**
     * 根据产品ID查询产品信息
     * @param productId 产品ID
     * @return 产品信息
     */
    public Optional<PbpProductDto> getByProductId(Long productId) {
        return productCaffeineCache.get(productId);
    }

    /**
     * 根据任务ID查询任务信息
     * @param taskId 任务ID
     * @return 任务信息
     */
    public Optional<PbpTaskDto> getTaskById(Long taskId) {
        return taskCaffeineCache.get(taskId);
    }

    /**
     * 根据媒体平台ID查询媒体平台信息
     * @param mediaPlatformId 媒体平台ID
     * @return 媒体平台信息
     */
    public Optional<PbpMediaPlatformDto> getByMediaPlatformId(Long mediaPlatformId) {
        return mediaPlatformCaffeineCache.get(mediaPlatformId);
    }

    public String getEventType(Long channelId, String eventType) {
        Optional<PutPlanDto> optional = getByChannelId(channelId);
        if (!optional.isPresent()) {
            log.info("not found channelId {}", channelId);
            return null;
        }
        PutPlanDto putPlan = optional.get();
        Long putPlanId = putPlan.getId();
        if (!putPlan.getIsCallBack()) {
            log.info("the callback configuration is not enabled：channelId is {}，planId is {}，eventType is {}",
                    channelId, putPlanId, eventType);
            return null;
        }
        Map<String, String> actionIdMap = ObjectUtils.defaultIfNull(putPlan.getActionIdMap(), Collections.emptyMap());
        if (!actionIdMap.containsKey(eventType)) {
            log.info("tuia event type not supported：channelId is {}，planId is {}，eventType is {}", channelId, putPlanId, eventType);
            return null;
        }
        if (StringUtils.isBlank(actionIdMap.get(eventType))) {
            log.info("media event type is not defined：channelId is {}，planId is {}，eventType is {}", channelId, putPlanId, eventType);
            return null;
        }
        //概率没命中 万分比
        int randomNumber = RandomUtils.nextInt(0, 10000);
        if (randomNumber >= putPlan.getProbabilityBack()) {
            log.info("概率回传未命中：channelId is {}，planId is {}，eventType is {}，PR is {}, randomNumber is {}",
                    channelId, putPlanId, eventType, putPlan.getProbabilityBack(), randomNumber);
            return null;
        }
        return actionIdMap.get(eventType);
    }

    public PbpEventCallbackStrategyContext getEventCallbackStrategyContext(Long channelId, String aEventType) {
        PbpEventCallbackStrategyContext context = new PbpEventCallbackStrategyContext();
        context.setChannelId(channelId);
        context.setAEventType(aEventType);
        context.setShouldCallback(Boolean.FALSE);

        Optional<PutPlanDto> optional = getByChannelId(channelId);
        if (!optional.isPresent()) {
            log.info("投放计划不存在：channelId={}", channelId);
            return context;
        }
        PutPlanDto putPlan = optional.get();
        Long putPlanId = putPlan.getId();

        context.setPlanId(putPlanId);

        if (!putPlan.getIsCallBack()) {
            log.info("该计划无需回传媒体数据：channelId={}，planId={}，aEventType={}", channelId, putPlanId, aEventType);
            return context;
        }

        PbpMediaCallbackStrategyDto mediaCallbackStrategyDto = putPlan.getMediaCallbackStrategyDto();
        List<PbpMediaCallbackStrategyDto.StrategyItem> strategySet = mediaCallbackStrategyDto.getStrategySet();
        PbpMediaCallbackStrategyDto.StrategyItem strategyItem = strategySet.get(0);
        Integer strategyType = strategyItem.getStrategyType();
        List<PbpMediaCallbackStrategyDto.StrategyContent> strategyRules = strategyItem.getStrategyGroup();

        context.setStrategyType(strategyType);

        for (PbpMediaCallbackStrategyDto.StrategyContent rule : strategyRules) {
            Map<String, String> eventTypeMap = ObjectUtils.defaultIfNull(rule.getEventTypeMap(), Collections.emptyMap());
            String mEventType = eventTypeMap.get(aEventType);
            if (StringUtils.isNotBlank(mEventType)) {
                context.setMEventType(mEventType);
                context.setStrategyRule(rule);
                break;
            }
        }

        if (Objects.isNull(context.getStrategyRule())) {
            log.warn("该计划媒体回传策略数据异常：channelId={}，planId={}，callbackStrategy={}", channelId, putPlanId, JSON.toJSONString(mediaCallbackStrategyDto));
            return context;
        }

        if (Objects.equals(strategyType, MediaCallbackStrategyConsts.StrategyType.TYPE_PROB)) {
            // 概率回传
            PbpMediaCallbackStrategyDto.StrategyContent strategyRule = context.getStrategyRule();
            int probVal = strategyRule.getCbProb();
            int randomNumber = RandomUtils.nextInt(0, 10000);
            if (randomNumber >= probVal) {
                log.info("概率回传未命中：channelId={}，planId={}，aEventType={}，probVal={}, randomNum={}", channelId, putPlanId, aEventType, probVal, randomNumber);
                return context;
            }
        } else if (Objects.equals(strategyType, MediaCallbackStrategyConsts.StrategyType.TYPE_AMOUNT)) {
            /*
            按收益回传，主流程中不同步回传媒体数据，在异步流程中回传
            这里什么也不做，在主流程中进行判断
             */
        } else {
            throw new RuntimeException("未知的回传策略类型" + strategyType);
        }

        context.setShouldCallback(Boolean.TRUE);
        return context;
    }

    /**
     * 根据渠道号查询对应的计划 不存在则返回空计划
     *
     * @param channelId
     * @return
     */
    private Optional<PutPlanDto> queryByChannelId(Long channelId) {
        PutPlanDto putPlanDto = remotePurchaseService.getPlanByChannelId(channelId);
        return Optional.ofNullable(putPlanDto);
    }

    private Optional<PbpProductDto> queryByProductId(Long productId) {
        PbpProductDto pbpProductDto = remotePurchaseService.getProductById(productId);
        return Optional.ofNullable(pbpProductDto);
    }

    private Optional<PbpTaskDto> queryByTaskId(Long taskId) {
        PbpTaskDto pbpTaskDto = remotePurchaseService.getTaskById(taskId);
        return Optional.ofNullable(pbpTaskDto);
    }

    private Optional<PbpMediaPlatformDto> queryByMediaPlatformId(Long mediaPlatformId) {
        PbpMediaPlatformDto pbpMediaPlatformDto = remotePurchaseService.getMediaPlatformById(mediaPlatformId);
        return Optional.ofNullable(pbpMediaPlatformDto);
    }

}

