package cn.com.duiba.tuia.risk.center.api.util;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ColorConvertOp;
import java.io.IOException;
import java.io.InputStream;

/*
 * function: 用汉明距离进行图片相似度检测的Java实现
 * pHash-like image hash.
 * Based On: http://www.hackerfactor.com/blog/index.php?/archives/432-Looks-Like-It.html
 * 参考：http://blog.csdn.net/sunhuaqiang1/article/details/70232679
 */
public class ImagePHashUtil {

    private ColorConvertOp colorConvert = new ColorConvertOp(ColorSpace.getInstance(ColorSpace.CS_GRAY), null);
    private int size = 32;
    private int smallerSize = 8;
    private double[] c;

    public ImagePHashUtil() {
        initCoefficients();
    }

    private static final ImagePHashUtil imagePHashUtil = new ImagePHashUtil();

    public static int distanceValue(String hashDegree1, String hashDegree2) {
        return imagePHashUtil.distance(hashDegree1, hashDegree2);
    }

    public static String getHashValue(InputStream is) throws IOException {
        return imagePHashUtil.getHash(is);
    }

    /***
     * @Description: distance 计算汉明距离
     *
     * @Param: [s1, s2]
     * @Return: int
     * @throws:
     * @author: youhaijun
     * @Date: 2017/10/13
     */
    public int distance(String s1, String s2) {
        int counter = 0;
        for (int k = 0; k < s1.length(); k++) {
            if (s1.charAt(k) != s2.charAt(k)) {
                counter++;
            }
        }
        return counter;
    }

    /***
     * @Description: getHash 获取图片phash
     *
     * @Param: [is]
     * @Return: binary string' (like. 001010111011100010) which is easy to do a hamming distance on
     * @throws:
     * @author: youhaijun
     * @Date: 2017/10/13
     */
    public String getHash(InputStream is) throws IOException {
        BufferedImage img = ImageIO.read(is);

        // 1. Reduce size(缩小尺寸).pHash以小图片开始，但图片大于8*8，32*32是最好的。这样做的目的是简化了DCT的计算，而不是减小频率。
        img = resize(img, size, size);

        // 2. Reduce color(简化色彩).将图片转化成灰度图像，进一步简化计算量
        img = grayscale(img);

        double[][] vals = new double[size][size];

        for (int x = 0; x < img.getWidth(); x++) {
            for (int y = 0; y < img.getHeight(); y++) {
                vals[x][y] = getBlue(img, x, y);
            }
        }
        img = null;
        // 3. Compute the DCT(计算DCT).计算图片的DCT变换，得到32*32的DCT系数矩阵
        double[][] dctVals = applyDCT(vals);

        // 4. Reduce the DCT.虽然DCT的结果是32*32大小的矩阵，但我们只要保留左上角的8*8的矩阵，这部分呈现了图片中的最低频率
        //5. Compute the average value.如同均值哈希一样，计算DCT的均值。
        double total;

        total = getTotal(dctVals);

        double avg = total / (double) ((smallerSize * smallerSize) - 1);

        // 6. Further reduce the DCT.这是最主要的一步，根据8*8的DCT矩阵，设置0或1的64位的hash值，大于等于DCT均值的设为”1”，小于DCT均值的设为“0”。组合在一起，就构成了一个64位的整数，这就是这张图片的指纹
        StringBuilder hash = new StringBuilder();

        for (int x = 0; x < smallerSize; x++) {
            for (int y = 0; y < smallerSize; y++) {
                if (x != 0 && y != 0) {
                    hash.append(dctVals[x][y] > avg ? "1" : "0");
                }
            }
        }
        dctVals = null;
//        System.gc();
        return hash.toString();
    }

    private double getTotal(double[][] dctVals) {
        double temp = 0;
        for (int x = 0; x < smallerSize; x++) {
            for (int y = 0; y < smallerSize; y++) {
                temp += dctVals[x][y];
            }
        }
        temp -= dctVals[0][0];
        return temp;
    }

    /***
     * Reduce size(缩小尺寸)
     */
    private BufferedImage resize(BufferedImage image, int width, int height) {
        BufferedImage resizedImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g = resizedImage.createGraphics();
        g.drawImage(image, 0, 0, width, height, null);
        g.dispose();
        return resizedImage;
    }


    /***
     * Reduce color(简化色彩)
     */
    private BufferedImage grayscale(BufferedImage img) {
        colorConvert.filter(img, img);
        return img;
    }

    /***
     * 获取灰度
     * @param img
     * @param x
     * @param y
     * @return
     */
    private int getBlue(BufferedImage img, int x, int y) {
        return (img.getRGB(x, y)) & 0xff;
    }


    private void initCoefficients() {
        c = new double[size];

        for (int i = 1; i < size; i++) {
            c[i] = 1;
        }
        c[0] = 1 / Math.sqrt(2.0);
    }

    private double[][] applyDCT(double[][] f) {
        int s = size;

        double[][] ff = new double[s][s];
        for (int u = 0; u < s; u++) {
            for (int v = 0; v < s; v++) {
                double sum = 0.0;
                for (int i = 0; i < s; i++) {
                    for (int j = 0; j < s; j++) {
                        sum += Math.cos(((2 * i + 1) / (2.0 * s)) * u * Math.PI) * Math.cos(((2 * j + 1) / (2.0 * s)) * v * Math.PI) * (f[i][j]);
                    }
                }
                sum *= ((c[u] * c[v]) / 4.0);
                ff[u][v] = sum;
            }
        }
        return ff;
    }

}