package cn.com.duiba.tuia.risk.center.api.common;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

/**
 * Basic Java Bean implementation of {@code Pageable}.
 *
 * @author ikongyi
 */
public class PageRequest implements Pageable, Serializable {

    private static final long serialVersionUID = 9078336764139449739L;

    public static final Integer DEFAULT_OFFSET = 0;
    public static final Integer DEFAULT_PAGE_SIZE = 200;

    private Integer offset;
    private Integer pageSize;
    private List<OrderBy> orderByList;

    /**
     * Creates a new unsorted {@link PageRequest}.
     *
     * @param offset zero-based page index, must not be negative.
     * @param size the size of the page to be returned, must be greater than 0.
     */
    public static PageRequest of(int offset, int size) {
        return of(offset, size, Collections.emptyList());
    }

    /**
     * Creates a new {@link PageRequest} with sort parameters applied.
     *
     * @param offset zero-based page index.
     * @param size the size of the page to be returned.
     * @param orderBy must not be {@literal null}
     */
    public static PageRequest of(int offset, int size, OrderBy orderBy) {
        return newPageRequest(offset, size, Collections.singletonList(orderBy));
    }

    /**
     * Creates a new {@link PageRequest} with sort parameters applied.
     *
     * @param offset zero-based page index.
     * @param size the size of the page to be returned.
     * @param orderByList must not be {@literal null}
     */
    public static PageRequest of(int offset, int size, List<OrderBy> orderByList) {
        return newPageRequest(offset, size, orderByList);
    }

    /**
     * Creates a new {@link PageRequest} with sort direction and properties applied.
     *
     * @param offset        zero-based page index, must not be negative.
     * @param size        the size of the page to be returned, must be greater than 0.
     * @param direction   must not be {@literal null}.
     * @param columnNames must not be {@literal null}.
     */
    public static PageRequest of(int offset, int size, OrderBy.Direction direction, String... columnNames) {
        return of(offset, size, OrderBy.by(direction, columnNames));
    }

    static PageRequest newPageRequest(int offset, int size, List<OrderBy> orderByList) {
        PageRequest pageRequest = new PageRequest();
        pageRequest.setOffset(offset);
        pageRequest.setPageSize(size);
        pageRequest.setOrderByList(orderByList);
        return pageRequest;
    }

    @Override
    public Integer getOffset() {
        return offset;
    }

    public void setOffset(Integer offset) {
        this.offset = offset;
    }

    @Override
    public Integer getPageSize() {
        return pageSize;
    }

    public void setPageSize(Integer pageSize) {
        this.pageSize = pageSize;
    }

    public void setOrderByList(List<OrderBy> orderByList) {
        this.orderByList = orderByList;
    }

    @Override
    public List<OrderBy> getSort() {
        return orderByList;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return String.format("Page request [offset %d, size: %s]", getOffset(), getPageSize());
    }
}

