package cn.com.duiba.tuia.ssp.center.api.tool;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * OCR识别工具
 * 文档地址：https://market.aliyun.com/products/57124001/cmapi027758.html
 */
@Slf4j
public class OCRUtil {

    /**
     * 发票API
     */
    private static final String INVOICE_URL = "https://ocrapi-invoice.taobao.com/ocrservice/invoice";

    /**
     * 发票API凭证
     */
    private static final String INVOICE_APP_CODE = "e6c12fd07875453cbe650280d86e1e4f";

    /**
     * 识别结果
     *
     * @param img 图像数据：base64编码，要求base64编码后大小不超过4M，最短边至少15px，最长边最大4096px，支持jpg/png/bmp格式，和url参数只能同时存在一个
     * @param url 图像url地址：图片完整URL，URL长度不超过1024字节，URL对应的图片base64编码后大小不超过4M，最短边至少15px，最长边最大4096px，支持jpg/png/bmp格式，和img参数只能同时存在一个
     * @return 识别结果
     */
    public static InvoiceResult getInvoiceResult(String img, String url) {
        // 校验参数
        if (StringUtils.isAllBlank(img, url)) {
            throw new RuntimeException("参数不能为空");
        }
        if (StringUtils.isNoneBlank(img, url)) {
            throw new RuntimeException("只能其中一个参数有值");
        }
        if (StringUtils.isBlank(img)) {
            img = StringUtils.EMPTY;
        }
        if (StringUtils.isBlank(url)) {
            url = StringUtils.EMPTY;
        }
        // 发起请求
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put("img", img);
        paramMap.put("url", url);
        RequestBody body = RequestBody.create(MediaType.get("application/json"), JSONObject.toJSONString(paramMap));
        Request request = new Request.Builder()
                .url(INVOICE_URL)
                .header("Authorization", "APPCODE " + INVOICE_APP_CODE)
                .header("Content-Type", "application/json; charset=UTF-8")
                .post(body)
                .build();

        OkHttpClient client = new OkHttpClient();
        Call call = client.newCall(request);
        try {
            Response response = call.execute();
            String result = Objects.requireNonNull(response.body()).string();
            log.info("OCR识别结果 result={}", result);
            JSONObject jsonObject = JSONObject.parseObject(result);
            if (StringUtils.isNotBlank(jsonObject.getString("error_code"))) {
                throw new RuntimeException(jsonObject.getString("error_msg"));
            }
            return JSONObject.parseObject(jsonObject.getString("data"), InvoiceResult.class);
        } catch (Exception e) {
            log.error("OCR识别异常", e);
            throw new RuntimeException(e.getMessage());
        }
    }

    @Setter
    @Getter
    @ToString
    public static class InvoiceResult {

        @JSONField(name = "发票代码")
        private String invoiceCode;

        @JSONField(name = "发票号码")
        private String invoiceNo;

        @JSONField(name = "开票日期")
        private String invoiceDate;

        @JSONField(name = "校验码")
        private String invoiceVerifyCode;

        @JSONField(name = "发票金额")
        private String invoiceAmount;

        @JSONField(name = "发票税额")
        private String invoiceTaxAmount;

        @JSONField(name = "不含税金额")
        private String invoiceNotIncludedTaxAmount;

        @JSONField(name = "受票方名称")
        private String customerName;

        @JSONField(name = "受票方税号")
        private String customerNo;

        @JSONField(name = "受票方地址、电话")
        private String customerAddressTel;

        @JSONField(name = "受票方开户行、账号")
        private String customerBankAccount;

        @JSONField(name = "销售方名称")
        private String sellerName;

        @JSONField(name = "销售方税号")
        private String sellerNo;

        @JSONField(name = "销售方地址、电话")
        private String sellerAddressTel;

        @JSONField(name = "销售方开户行、账号")
        private String sellerBankAccount;
    }
}
