package cn.com.duiba.tuia.ssp.center.api.tool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * @author fangzhipeng
 * @date 2018/7/26
 */
public class DateUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtil.class);

    public static final String DEFAULT_FORMATTER = "yyyy-MM-dd HH:mm:ss";

    /**
     * 获取两个日期之间的日期(包含起止日期)
     * eg: 2018-07-10 ~ 2018-07-15 返回 2018-07-10 2018-07-11 2018-07-12 2018-07-13 2018-07-14 2018-07-15
     * @param startDateStr 开始日期
     * @param endDateStr 结束日期
     * @return 日期列表
     */
    public static List<String> getBetweenDate(String startDateStr, String endDateStr) {
        try {
            DateTimeFormatter pattern = DateTimeFormatter.ofPattern("yyyy-MM-dd");
            LocalDate startDate = LocalDate.parse(startDateStr, pattern);
            LocalDate endDate = LocalDate.parse(endDateStr, pattern);
            List<String> result = new ArrayList<>();
            while (startDate.compareTo(endDate) <= 0) {
                result.add(startDate.format(pattern));
                startDate = startDate.plusDays(1);
            }
            return result;
        } catch (Exception e) {
            LOGGER.error("日期解析异常，startDate:" + startDateStr + ", endDate:" + endDateStr, e);
            return Collections.emptyList();
        }
    }

    public static List<String> getBetweenDate(Date startDate, Date endDate) {
        DateTimeFormatter pattern = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        LocalDateTime localStartDateStr = startDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
        LocalDateTime localEndDateStr = endDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
        List<String> result = new ArrayList<>();
        while (localStartDateStr.compareTo(localEndDateStr) <= 0) {
            result.add(localStartDateStr.format(pattern));
            localStartDateStr = localStartDateStr.plusDays(1);
        }
        return result;
    }

    /**
     * 获取yyyy-MM-dd 类型的日期
     * @param date 要获取日期
     * @return 对应字符串
     */
    public static String YYYY_MM_DD(LocalDate date) {
        return YYYY_MM_DD(date, 0);
    }

    /**
     * 获取yyyy-MM-dd 类型的日期
     * @param date 要获取日期
     * @param add 对应日期的加减天数
     * @return 对应字符串
     */
    public static String YYYY_MM_DD(LocalDate date, int add) {
        String pattern = "yyyy-MM-dd";
        return getDateString(date, add, pattern);
    }

    /**
     * 获取yyyy-MM-dd 类型的日期
     * @param date 要获取日期
     * @return 对应字符串
     */
    public static String YYYYMMDD(LocalDate date) {
        return YYYYMMDD(date, 0);
    }

    /**
     * 获取yyyyMMdd 类型的日期
     * @param date 要获取日期
     * @param add 对应日期的加减天数
     * @return 对应字符串
     */
    public static String YYYYMMDD(LocalDate date, int add) {
        String pattern = "yyyyMMdd";
        return getDateString(date, add, pattern);
    }

    /**
     * 将date 转换为对应的str
     * @param date 日期
     * @param add 加减date 后的日期
     * @param pattern 对应的转换成的格式 eg：yyyy-MM-dd 2018-08-27
     * @return 对应格式转换后的字符串
     */
    private static String getDateString(LocalDate date, int add, String pattern) {
        return date.plusDays(add).format(DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 将date 转换为对应的str
     * @param date 日期
     * @param add 加减date 后的日期
     * @return 对应格式转换后的Date
     */
    public static Date getDate(LocalDate date, int add) {
        LocalDate localDate = date.plusDays(add);
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 获取当前时间与当天最晚时间的差值（秒数）
     * @return 对应格式转换后的Date
     */
    public static int getTodayResidualSeconds() {
        LocalDateTime todayEnd = LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
        Long seconds = Duration.between(LocalDateTime.now(), todayEnd).toMillis() / 1000;
        return seconds.intValue();
    }


    /**
     * 获取当前日期往前第N天日期，格式:yyyy-MM-dd
     *
     * @return
     * @throws Exception
     */
    public static Date getNearlyNDay(int i) {
        SimpleDateFormat dft = new SimpleDateFormat("yyyy-MM-dd");
        Calendar date = Calendar.getInstance();
        date.setTime(new Date());
        date.set(Calendar.DATE, date.get(Calendar.DATE) - i);
        Date endDate = null;
        try {
            endDate = dft.parse(dft.format(date.getTime()));
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return endDate;
    }


}
