package cn.com.duiba.tuia.enginex.component.support;

import cn.com.duiba.boot.utils.AopTargetUtils;
import cn.com.duiba.tuia.enginex.api.component.Component;
import cn.com.duiba.tuia.enginex.api.component.ComponentChain;
import cn.com.duiba.tuia.enginex.api.component.ComponentRequest;
import cn.com.duiba.tuia.enginex.api.component.ComponentResponse;
import cn.com.duiba.tuia.enginex.api.component.ComponentStopException;
import cn.com.duiba.tuia.enginex.api.component.annotation.ComponentConfig;
import cn.com.duiba.tuia.enginex.api.component.annotation.Ordered;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.annotation.PostConstruct;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author lijicong
 * @since 2021-03-15
 */
@Slf4j
public abstract class AbstractComponentChain implements ComponentChain, ApplicationContextAware {

    protected ApplicationContext applicationContext;

    protected List<Component> components;

    @Override
    public List<Component> getComponents() {
        return components;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @PostConstruct
    public void init() {
        Map<String, Component> componentBeansMap = applicationContext.getBeansOfType(Component.class);
        List<Component> components = componentBeansMap.values().stream()
                .filter(item -> !(item instanceof ComponentChain)).collect(Collectors.toList());
        components.sort(Comparator.comparing(this::getOrder));
        this.components = components;
    }

    private int getOrder(Component component) {
        Component target = component;
        if (AopUtils.isAopProxy(component)) {
            try {
                target = AopTargetUtils.getTarget(component);
            } catch (Exception e) {
                log.error("代理类获取目标类失败");
                return Ordered.HIGHEST_PRECEDENCE;
            }
        }
        Class<? extends Component> clazz = target.getClass();
        ComponentConfig annotation = clazz.getAnnotation(ComponentConfig.class);
        if (Objects.isNull(annotation)) {
            return Ordered.HIGHEST_PRECEDENCE;
        }
        return annotation.order();
    }

    @Override
    public void execute(ComponentRequest request, ComponentResponse response) {
        List<Component> components = getComponents();

        for (Component component : components) {
            try {
                component.execute(request, response);
            } catch (ComponentStopException e) {
                log.warn("组件执行终止 request={}, response={}", request, response, e);
                break;
            } catch (Exception e) {
                log.warn("组件执行异常 request={}, response={}", request, response, e);
            }
        }

    }
}
