package cn.com.duiba.tuia.union.star.center.api.cache.service.impl;

import cn.com.duiba.tuia.union.star.center.api.cache.service.RemoteDeviceUaParseCache;
import cn.com.duiba.tuia.union.star.center.api.cache.service.utils.UaSplitUtils;
import cn.com.duiba.tuia.union.star.center.api.remoteservice.RemoteDeviceUaParseService;
import cn.com.duiba.tuia.union.star.center.api.remoteservice.domain.rsp.MobileUaInfoDTO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 设备ua解析服务缓存
 *
 * @author miaohongshuai
 * @date 2021-07-12
 */
@Slf4j
@Service
public class RemoteDeviceUaParseCacheImpl implements RemoteDeviceUaParseCache {
    @Autowired
    private ExecutorService executorService;
    @Resource
    private RemoteDeviceUaParseService remoteDeviceUaParseService;

    private LoadingCache<String, Optional<MobileUaInfoDTO>> MOBILE_UA_CACHE;

    /**
     * 初始化本地缓存
     */
    @PostConstruct
    public void init() {
        MOBILE_UA_CACHE = CacheBuilder.newBuilder()
                .maximumSize(10000)
                .concurrencyLevel(20)
                .refreshAfterWrite(US_EXPIRE_REFRESH_SECONDS, TimeUnit.SECONDS)
                .build(MOBILE_UA_CACHE_LOADER);
    }

    /**
     * 【没有热刷新】机制，只能客户端application.properties配置，或者apollo配置
     * 过期刷新策略
     * 过期删除策略
     */
    @Value("${us.expire.refresh.seconds:600}")
    public Integer US_EXPIRE_REFRESH_SECONDS;

    private CacheLoader<String, Optional<MobileUaInfoDTO>> MOBILE_UA_CACHE_LOADER = new CacheLoader<String, Optional<MobileUaInfoDTO>>() {
        @Override
        public Optional<MobileUaInfoDTO> load(String phoneModel) {
            if (StringUtils.isEmpty(phoneModel)) {
                return Optional.empty();
            }
            MobileUaInfoDTO mobileUaInfoDTO = remoteDeviceUaParseService.selectUaParseInfoByPhoneModel(phoneModel);
            return Optional.ofNullable(mobileUaInfoDTO);
        }

        @Override
        public ListenableFuture<Optional<MobileUaInfoDTO>> reload(String keyStr, Optional<MobileUaInfoDTO> oldValue) {
            ListenableFutureTask<Optional<MobileUaInfoDTO>> task = ListenableFutureTask.create(() -> {
                Optional<MobileUaInfoDTO> result;
                try {
                    result = load(keyStr);
                } catch (Exception e) {
                    log.error("MOBILE_UA_CACHE_LOADER reload e", e);
                    return oldValue;
                }
                return result;
            });
            executorService.submit(task);
            return task;
        }
    };

    @Override
    public MobileUaInfoDTO getMobileInfoByUa(String ua) {
        if (StringUtils.isBlank(ua)) {
            return null;
        }
        Optional<MobileUaInfoDTO> mobileUaInfoDTO = null;
        try {
            String phoneModel = UaSplitUtils.parseUaToPhoneModel(ua.toLowerCase());
            if (StringUtils.isBlank(phoneModel)) {
                return null;
            }
            mobileUaInfoDTO = MOBILE_UA_CACHE.get(phoneModel);
        } catch (ExecutionException e) {
            log.warn("MOBILE_UA_CACHE get error.ua={}", ua, e);
            return null;
        }
        return mobileUaInfoDTO.orElse(null);
    }

    @Override
    public MobileUaInfoDTO getMobileInfoByPhoneModel(String phoneModel) {
        if (StringUtils.isBlank(phoneModel)) {
            return null;
        }
        Optional<MobileUaInfoDTO> mobileUaInfoDTO = null;
        try {
            mobileUaInfoDTO = MOBILE_UA_CACHE.get(phoneModel.toLowerCase());
        } catch (ExecutionException e) {
            e.printStackTrace();
        }
        return mobileUaInfoDTO.orElse(null);
    }
}
