package cn.com.tuia.advert.message.consumer;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.core.RedisTemplate;

import com.alibaba.ttl.threadpool.TtlExecutors;

import cn.com.tuia.advert.message.RedisMessageChannel;

/**
 * redis消息接受
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: RedisMsgSubscriber.java , v 0.1 2017/3/16 下午8:45 ZhouFeng Exp $
 */
public class RedisMsgSubscriber implements MessageListener, ApplicationContextAware {

    private static Logger                                        log        = LoggerFactory.getLogger(RedisMsgSubscriber.class);

    private final ExecutorService                                pool       = TtlExecutors.getTtlExecutorService(Executors.newFixedThreadPool(5));

    private Map<RedisMessageChannel, AbstractRedisMessageHandle> handlesMap = new HashMap<>();

    public final Logger                                          logger     = LoggerFactory.getLogger(getClass());

    private Set<String>                                          channels   = new HashSet<>();

    private RedisTemplate<String, String>                        redisTemplate;

    public RedisMsgSubscriber(RedisTemplate<String, String> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    public RedisMsgSubscriber(RedisTemplate<String, String> redisTemplate,
                              Map<RedisMessageChannel, ? extends AbstractRedisMessageHandle> handlesMap) {
        this(redisTemplate);
        this.handlesMap.putAll(handlesMap);
        Set<RedisMessageChannel> redisMessageChannels = handlesMap.keySet();
        for (RedisMessageChannel channel : redisMessageChannels) {
            channels.add(channel.getChannel());
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        // 从IOC中取出所有AbstractRedisMessageHandle对象，根据其注解建立与消息的映射关系
        Map<String, AbstractRedisMessageHandle> map = applicationContext.getBeansOfType(AbstractRedisMessageHandle.class);
        Collection<AbstractRedisMessageHandle> handles = map.values();
        for (AbstractRedisMessageHandle handle : handles) {
            RedisMessageHandle annotation = handle.getClass().getAnnotation(RedisMessageHandle.class);
            if (annotation == null) {
                continue;
            }
            RedisMessageChannel channel = annotation.value();
            channels.add(channel.getChannel());
            handlesMap.put(channel, handle);
        }
    }

    @Override
    public void onMessage(Message message, byte[] pattern) {

        byte[] body = message.getBody();// 请使用valueSerializer
        byte[] channel = message.getChannel();
        // 请参考配置文件，本例中key，value的序列化方式均为string。
        // 其中key必须为stringSerializer。和redisTemplate.convertAndSend对应
        String msgContent = (String) redisTemplate.getValueSerializer().deserialize(body);
        String topic = redisTemplate.getStringSerializer().deserialize(channel);
        log.info(topic + ";msg=" + message);

        AbstractRedisMessageHandle handle = null;

        RedisMessageChannel channelEnum = RedisMessageChannel.getByChannel(topic);

        if (channelEnum == null) {
            logger.error("channel:[{}] not dedeclared", channel);
            return;
        }

        if (handlesMap != null && !handlesMap.isEmpty()) {
            handle = handlesMap.get(channelEnum);
        }

        if (handle == null) {
            logger.error("not find handle of channel:[{}]", channel);
            return;
        }

        pool.submit(handle.createHandle(channelEnum, msgContent));
    }
}
