package cn.com.tuia.advert.stream;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * java stream 工具集
 *
 * @author qiuqiyang
 * @date 2018-01-24 16:31
 */
public class StreamUtils {

    /**
     * List 根据某个字段去重
     *
     * @param keyExtractor
     *            字段
     * @param <T>
     *            类型
     * @return 判断值
     */
    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Set<Object> seen = ConcurrentHashMap.newKeySet();
        return t -> seen.add(keyExtractor.apply(t));
    }

    /**
     * 快速将一个集合的某个key提取出来
     *
     * @param collection
     *            对象集合
     * @param keyExtractor
     *            需要提取的key的引用
     * @param <T>
     *            需提取对象集合
     * @param <R>
     *            返回类型
     * @return 对应结果列表
     */
    public static <T, R> List<R> toList(Collection<T> collection, Function<? super T, R> keyExtractor) {
        return toList(collection, keyExtractor, null,null);
    }
    /**
     * 快速将一个集合的某个key提取出来
     *
     * @param collection
     *            对象集合
     * @param keyExtractor
     *            需要提取的key的引用
     * @param <T>
     *            需提取对象集合
     * @param <R>
     *            返回类型
     * @return 对应结果列表
     */
    public static <T, R> Set<R> toSet(Collection<T> collection, Function<? super T, R> keyExtractor) {
        return toSet(collection, keyExtractor, null);
    }

    /**
     * 快速将一个集合的某个key提取出来
     *
     * @param collection
     *            对象集合
     * @param keyExtractor
     *            需要提取的key的引用
     * @param <T>
     *            需提取对象集合
     * @param <R>
     *            返回类型
     * @return 对应结果列表
     */
    public static <T, R> List<R> toList(Collection	<T> collection, Function<? super T, R> keyExtractor,
                                        Predicate<? super T> predicate,Boolean isDistinct) {
        Stream<T> stream = collection.stream();
        if (predicate != null) {
            stream = stream.filter(predicate);
        }
        if (isDistinct != null && isDistinct) {
            return stream.map(keyExtractor).distinct().collect(Collectors.toList());
        }
        return stream.map(keyExtractor).collect(Collectors.toList());
    }
    /**
     * 快速将一个集合的某个key提取出来,keyExtractor去重
     *
     * @param collection
     *            对象集合
     * @param keyExtractor
     *            需要提取的key的引用
     * @param <T>
     *            需提取对象集合
     * @param <R>
     *            返回类型
     * @return 对应结果列表
     */
    public static <T, R> List<R> toDistinctList(Collection<T> collection, Function<? super T, R> keyExtractor) {
        return toList(collection, keyExtractor, null,true);
    }
    /**
     * 快速将一个集合的某个key提取出来
     *
     * @param collection
     *            对象集合
     * @param keyExtractor
     *            需要提取的key的引用
     * @param <T>
     *            需提取对象集合
     * @param <R>
     *            返回类型
     * @return 对应结果列表
     */
    public static <T, R> Set<R> toSet(Collection	<T> collection, Function<? super T, R> keyExtractor,
                                      Predicate<? super T> predicate) {
        Stream<T> stream = collection.stream();
        if (predicate != null) {
            stream = stream.filter(predicate);
        }
        return stream.map(keyExtractor).collect(Collectors.toSet());
    }

    /**
     * 解决原Stream流中集合转Map，value为空报空指针异常的问题
     *  {@code
     *         List<Test> testList = new ArrayList<>();
     *         testList.add(new Test("222232323",150L));
     *         testList.add(new Test("222232324",null));
     *         Map<String,Long> map = testList.stream().collect(Collectors.toMap(Test::getMemberId,Test::getSchoolId));
     *  }
     * {@code
     *       Exception in thread "main" java.lang.NullPointerException
     * 	       			at java.util.HashMap.merge(HashMap.java:1225)
     * 					at java.util.stream.Collectors.lambda$toMap$58(Collectors.java:1320)
     * 					at java.util.stream.ReduceOps$3ReducingSink.accept(ReduceOps.java:169)
     * 					at java.util.ArrayList$ArrayListSpliterator.forEachRemaining(ArrayList.java:1382)
     * 					at java.util.stream.AbstractPipeline.copyInto(AbstractPipeline.java:481)
     * 					at java.util.stream.AbstractPipeline.wrapAndCopyInto(AbstractPipeline.java:471)
     * 					at java.util.stream.ReduceOps$ReduceOp.evaluateSequential(ReduceOps.java:708)
     * 					at java.util.stream.AbstractPipeline.evaluate(AbstractPipeline.java:234)
     * 					at java.util.stream.ReferencePipeline.collect(ReferencePipeline.java:499)
     * }
     * @param collection
     * @param keyMapper a mapping function to produce keys
     * @param valueMapper a mapping function to produce values
     * @param <T>  the type of the input elements
     * @param <K>  the output type of the key mapping function
     * @param <U> the output type of the value mapping function
     * @return
     */
    public static  <T, K, U> Map<K, U> toMap(Collection<T> collection, Function<? super T, ? extends K> keyMapper,
                                             Function<? super T, ? extends U> valueMapper) {
        Objects.requireNonNull(collection);
        Stream<T> stream = collection.stream();
        BiConsumer<Map<K, U>, T> accumulator
                = (map, element) -> map.put(keyMapper.apply(element),
                valueMapper.apply(element));
        BiConsumer<Map<K, U>, Map<K, U>> combiner = (a,b) ->  a.putAll(b);
        return stream.collect(HashMap::new,accumulator,combiner);
    }
}