package cn.com.duiba.wolf.cache;

import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 缓存客户端接口
 */
public interface CacheClient {

    /**
     * 根据Key读取Cache中的值
     * 
     * @param key
     * @return
     */
	public boolean add(String key, int exp, Object value) throws Exception;
	
    <T> T get(String key);

    public long getLong(String key);

    /**
     * 设置值,如果value为null不会放入缓存
     * 
     * @param key
     * @param value
     * @param expSeconds 过期时间（单位：秒）
     */
    boolean set(String key, Object value, int expSeconds);

    /**
     * 设置值,如果value为null不会放入缓存
     * 
     * @param key
     * @param value
     * @param exp
     * @param timeUnit
     */
    boolean set(String key, Object value, int exp, TimeUnit timeUnit);

    /**
     * 根据Key读取Cache中的值,当值为null时会自动调用cacheLoader.load方法获得值并set进缓存
     *
     * @param key
     * @param exp
     * @param timeUnit
     * @param isCacheNull isCacheNull为true时, 如果cacheLoader.load得到null,则会把NullCache.NULL放入缓存,以减少对数据库的穿透,get时会判断如果是NullCache实例则直接返回null
     * @param cacheLoader
     * @param <T>
     * @return
     */
    <T> T getWithCacheLoader(String key, int exp, TimeUnit timeUnit, boolean isCacheNull, CacheLoader<T> cacheLoader);

    /**
     * 根据Key读取Cache中的值,当值为null时会自动调用cacheLoader.load方法获得值并set进缓存
     *
     * @param key
     * @param exp
     * @param timeUnit
     * @param cacheLoader
     * @param <T>
     * @return
     */
    <T> T getWithCacheLoader(String key, int exp, TimeUnit timeUnit, CacheLoader<T> cacheLoader);

    /**
     * 移除key
     * 
     * @param key
     * @return
     */
    boolean remove(String key);

    /**
     * 清空所有缓存
     */
    void flushAll();

    // /**
    // * 是否包含key
    // *
    // * @param key
    // * @return
    // */
    // boolean containsKey(String key);

    public long incr(String key, long by);

    public long incr(String key, long by, long exp, TimeUnit timeUnit);

    public long incr(String key, long delta, long initValue, long expiry, TimeUnit unit);

    public long decr(String key, long by);

    public long decr(String key, long by, long exp, TimeUnit timeUnit);

    public long decr(String key, long delta, long initValue, long expiry, TimeUnit unit);

    /**
     * CAS操作
     * @param key key必须已经在缓存中存在,否则会返回false
     * @param expected 期望的值
     * @param value 要替换的值
     * @param expSeconds 缓存超时时间
     * @return
     */
    public boolean cas(String key, Object expected, Object value, int expSeconds);

}
