package cn.lili.common.redisExt;

import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import redis.clients.jedis.JedisPoolConfig;

import java.time.Duration;

/**
 * @Author yuquanfeng
 * @Date 2023/5/26
 * @Description
 * @Param
 * @return
 **/
public class JedisConnectionConfiguration {

    private final RedisExtProperties properties;

    JedisConnectionConfiguration(RedisExtProperties properties){
        this.properties = properties;
    }

    public JedisConnectionFactory redisConnectionFactory() {
        return createJedisConnectionFactory();
    }

    private JedisConnectionFactory createJedisConnectionFactory() {
        JedisClientConfiguration clientConfiguration = getJedisClientConfiguration();
        return new JedisConnectionFactory(getStandaloneConfig(), clientConfiguration);
    }

    protected final RedisStandaloneConfiguration getStandaloneConfig() {
        RedisStandaloneConfiguration config = new RedisStandaloneConfiguration();
        config.setHostName(this.properties.getHost());
        config.setPort(this.properties.getPort());
        config.setPassword(RedisPassword.of(this.properties.getPassword()));
        config.setDatabase(this.properties.getDatabase());
        return config;
    }


    private JedisClientConfiguration getJedisClientConfiguration() {
        JedisClientConfiguration.JedisClientConfigurationBuilder builder = applyProperties(
                JedisClientConfiguration.builder());
        RedisExtProperties.Pool pool = this.properties.getPool();
        if (pool != null) {
            applyPooling(pool, builder);
        }
        return builder.build();
    }

    private void applyPooling(RedisExtProperties.Pool pool,
                              JedisClientConfiguration.JedisClientConfigurationBuilder builder) {
        builder.usePooling().poolConfig(jedisPoolConfig(pool));
    }


    private JedisPoolConfig jedisPoolConfig(RedisExtProperties.Pool pool) {
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxTotal(pool.getMaxActive());
        config.setMaxIdle(pool.getMaxIdle());
        config.setMinIdle(pool.getMinIdle());
        if (pool.getMaxWait() != null) {
            config.setMaxWait(Duration.ofMillis(pool.getMaxWait().toMillis()));
        }
        config.setTestWhileIdle(true);
        config.setTimeBetweenEvictionRunsMillis(90000);

        config.setSoftMinEvictableIdleTimeMillis(180000);
        config.setMinEvictableIdleTimeMillis(-1);
        config.setNumTestsPerEvictionRun(-1);

        return config;
    }


    private JedisClientConfiguration.JedisClientConfigurationBuilder applyProperties(
            JedisClientConfiguration.JedisClientConfigurationBuilder builder) {
        if (this.properties.isSsl()) {
            builder.useSsl();
        }
        if (this.properties.getTimeout() != null) {
            Duration timeout = this.properties.getTimeout();
            builder.readTimeout(timeout).connectTimeout(timeout);
        }
        return builder;
    }
}
