package cn.lili.common.threadpool;

import cn.lili.common.threadpool.factory.ThreadPoolFactoryBean;
import cn.lili.common.threadpool.properties.ThreadPoolConfig;
import cn.lili.common.threadpool.properties.ThreadPoolProperties;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.assertj.core.util.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.core.env.PropertyResolver;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @Author yuquanfeng
 * @Date 2023/5/25
 * @Description
 * @Param
 * @return
 **/
@Component
public class ThreadPoolBeanFactoryPostProcessor implements BeanDefinitionRegistryPostProcessor, EnvironmentAware {

    private Logger logger = LoggerFactory.getLogger(ThreadPoolBeanFactoryPostProcessor.class);

    private static final String THREAD_POOL_PREFIX = "duiba.threadpool";

    private BeanDefinitionRegistry beanDefinitionRegistry;

    private Set<String> threadPoolNames = Sets.newHashSet();

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry beanDefinitionRegistry) throws BeansException {
        this.beanDefinitionRegistry = beanDefinitionRegistry;
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory configurableListableBeanFactory) throws BeansException {
        if(threadPoolNames.isEmpty()){
            return;
        }
        for(String threadPoolName:threadPoolNames){

            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(ThreadPoolFactoryBean.class);
            builder.addPropertyValue("threadPoolName",threadPoolName);

            BeanDefinition beanDefinition = builder.getBeanDefinition();

            Boolean isPrimary = StringUtils.equals(threadPoolName, ThreadPoolProperties.DEFAULT_BEAN_NAME);

            beanDefinition.setPrimary(isPrimary);

            String beanName = threadPoolName;
            if(!beanName.endsWith(ThreadPoolProperties.BEAN_NAME_POSTFIX) && !isPrimary){
                beanName += ThreadPoolProperties.BEAN_NAME_POSTFIX;
            }

            beanDefinitionRegistry.registerBeanDefinition(beanName,beanDefinition);
        }
    }

    @Override
    public void setEnvironment(Environment environment) {

        ThreadPoolConfig config = Binder.get(environment).bindOrCreate(THREAD_POOL_PREFIX, ThreadPoolConfig.class);

        if(!config.getEnabled()){
            return;
        }

        threadPoolNames.add(ThreadPoolProperties.DEFAULT_BEAN_NAME);

        Map<String,ThreadPoolProperties> extra = config.getExtra();
        threadPoolNames.addAll(extra.keySet());

        if(config.getMaxSizeCount() >= ThreadPoolConfig.THREAD_MAX_SIZE){
            logger.warn("线程总配置数过高", "阁下的系统线程池中的线程总数配置过高，可能导致CPU线程上下文切换频繁，影响性能，建议降低配置总数，或者进行系统拆分，以降低线程池数目");
        }
    }
}
