package cn.lili.rocketmq.send;

import cn.lili.common.enums.ResultCode;
import cn.lili.common.exception.ServiceException;
import cn.lili.common.utils.StringUtils;
import cn.lili.rocketmq.RocketmqSendCallback;
import cn.lili.rocketmq.RocketmqSendCallbackBuilder;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @Author yuquanfeng
 * @Date 2023/5/25
 * @Description
 * @Param
 * @return
 **/
@Component
@Slf4j
public class RocketmqProduce {

    private static final String COLON = ":";

    @Autowired
    private RocketMQTemplate rocketMQTemplate;

    /**
     * 同步发送mq消息
     * @param topic 消息主题
     * @param tag 消息tag
     * @param message 消息体
     */
    public void sendMessageSync(String topic, String tag, String message){
        if(StringUtils.isBlank(topic) || StringUtils.isBlank(tag)){
            throw new ServiceException(ResultCode.PARAMS_ERROR);
        }
        String destination = topic + COLON + tag;
        try {
            SendResult sendResult = rocketMQTemplate.syncSend(destination, message);
            log.info("sendMessageSync topic:{},tag:{}, message:{}, result:{}", topic, tag, message, JSON.toJSON(sendResult));
        }catch (Exception e){
            log.error("sendMessageSync topic:{}, tag:{}, message:{}, e", topic, tag, message, e);
        }
    }

    /**
     * 同步发送消息（带超时时间）
     * @param topic 消息主题
     * @param tag 消息tag
     * @param message 消息体
     * @param timeout 超时时间（单位ms）
     */
    public void sendMessageSyncWithTimeout(String topic, String tag, String message, long timeout){
        if(StringUtils.isBlank(topic) || StringUtils.isBlank(tag)){
            throw new ServiceException(ResultCode.PARAMS_ERROR);
        }
        String destination = topic + COLON + tag;
        try {
            SendResult sendResult = rocketMQTemplate.syncSend(destination, message, timeout);
            log.info("sendMessageSyncWithTimeout topic:{},tag:{}, message:{}, result:{}", topic, tag, message, JSON.toJSON(sendResult));
        }catch (Exception e){
            log.error("sendMessageSyncWithTimeout topic:{}, tag:{}, message:{}, e", topic, tag, message, e);
        }
    }


    /**
     * 异步发送消息
     * @param topic
     * @param tag
     * @param message
     */
    public void sendMessageAsync(String topic, String tag, String message){
        if(StringUtils.isBlank(topic) || StringUtils.isBlank(tag)){
            throw new ServiceException(ResultCode.PARAMS_ERROR);
        }
        String destination = topic + COLON + tag;
        try {
            RocketmqSendCallback rocketmqSendCallback = RocketmqSendCallbackBuilder.commonCallback();
            rocketmqSendCallback.setTopic(topic);
            rocketmqSendCallback.setTag(tag);
            rocketmqSendCallback.setMessage(message);
            rocketMQTemplate.asyncSend(destination, message, rocketmqSendCallback);
            log.info("sendMessageAsync topic:{},tag:{}, message:{}, result:{}", topic, tag, message);
        }catch (Exception e){
            log.error("sendMessageAsync topic:{}, tag:{}, message:{}, e", topic, tag, message, e);
        }
    }

    /**
     * 异步发送消息（带超时时间 同时可以指定回调逻辑）
     * @param topic 消息主题
     * @param tag 消息tag
     * @param message 消息内容
     * @param sendCallback 回调逻辑
     * @param timeout 超时时间
     */
    public void sendMessageAsyncWithTimeout(String topic, String tag, String message, SendCallback sendCallback, long timeout){
        if(StringUtils.isBlank(topic) || StringUtils.isBlank(tag)){
            throw new ServiceException(ResultCode.PARAMS_ERROR);
        }
        String destination = topic + COLON + tag;
        try {
            rocketMQTemplate.asyncSend(destination, message, sendCallback, timeout);
            log.info("sendMessageAsyncWithTimeout topic:{},tag:{}, message:{}, result:{}", topic, tag, message);
        }catch (Exception e){
            log.error("sendMessageAsyncWithTimeout topic:{}, tag:{}, message:{}, e", topic, tag, message, e);
        }
    }

}
