package cn.tuia.explore.center.util;

import cn.tuia.explore.center.constant.GlobalConstant;
import cn.tuia.explore.center.entity.PureContent;
import cn.tuia.explore.center.enums.PushType;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.gexin.rp.sdk.base.IBatch;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.IQueryResult;
import com.gexin.rp.sdk.base.impl.AppMessage;
import com.gexin.rp.sdk.base.impl.ListMessage;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.TagMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.notify.Notify;
import com.gexin.rp.sdk.base.uitls.AppConditions;
import com.gexin.rp.sdk.dto.GtReq;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.google.common.base.Splitter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author ElinZhou zhoufeng@duiba.com.cn
 * @version $Id: GeTuiPush.java , v 0.1 2019-05-22 10:44 ElinZhou Exp $
 */
public class GeTuiPush {

    /**
     * logger
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(GeTuiPush.class);

    private static final int MAX_RETRY_TIMES = 3;

    private static final String GETUI_RESULT = "result";

    private static final String GETUI_RESULT_CODE_CID = "ok";

    private static final List<String> GETUI_SETTAG_RESULT = Arrays.asList("Success", "TagNoChange");
    /**
     * 个推每个用户打标签的的最大量
     */
    private static final Integer MAX_TAG_COUNT = 100;

    private IGtPush push;

    private String appId;

    private String appKey;

    private String packageName;

    private String component;

    public GeTuiPush(IGtPush push, String appId, String appKey, String packageName, String component) {
        this.push = push;
        this.appId = appId;
        this.appKey = appKey;
        this.packageName = packageName;
        this.component = component;
    }

    /**
     * 给cid设置标签
     *
     * @param clientId
     * @param adds     需要增加的标签
     * @param deletes  需要删除的标签
     */
    public void setTag(String clientId, Set<Long> adds, Set<Long> deletes) {
        Set<String> addTags = adds == null ? Collections.emptySet() :
                adds.stream().map(String::valueOf).collect(Collectors.toSet());
        Set<String> deleteTags = deletes == null ? Collections.emptySet() :
                deletes.stream().map(String::valueOf).collect(Collectors.toSet());

        IPushResult iPushResult = push.getUserTags(appId, clientId);

        if (null == iPushResult || null == iPushResult.getResponse() || null == iPushResult.getResponse().get("tags")) {
            return;
        }
        String tagStrings = iPushResult.getResponse().get("tags").toString();
        List<String> currentTagIds = Splitter.on(",").omitEmptyStrings().trimResults().splitToList(tagStrings);
        List<String> tags = new ArrayList<>(currentTagIds);
        //增加tag
        tags.addAll(addTags);
        //移除tag
        tags = tags.stream().filter(t -> !deleteTags.contains(t)).limit(MAX_TAG_COUNT).collect(Collectors.toList());

        IQueryResult iQueryResult = push.setClientTag(appId, clientId, tags);
        if ((null == iQueryResult || !GETUI_SETTAG_RESULT.contains(iQueryResult.getResponse().get(GETUI_RESULT).toString())) && LOGGER.isWarnEnabled()) {
            LOGGER.warn("请求个推setClientTag接口失败，返回参数：{}", JSON.toJSONString(iQueryResult));
        }

    }

    /**
     * 给指定标签推送内容
     *
     * @param pureContent
     * @param tagId
     */
    public void pushToTag(PureContent pureContent, long tagId) {
        TransmissionTemplate transmissionTemplate = buildTemplate(pureContent);
        TagMessage message = new TagMessage();
        message.setAppIdList(Collections.singletonList(appId));
        message.setTag(String.valueOf(tagId));
        message.setData(transmissionTemplate);
        // 设置消息离线，并设置离线时间
        message.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(TimeUnit.DAYS.toMillis(1));
        IPushResult iPushResult = push.pushTagMessage(message);
        handleResult(iPushResult);
    }


    /**
     * 推送给单个用户
     *
     * @param pureContent
     * @param clientIds
     */
    public void pushToUser(PureContent pureContent, Collection<String> clientIds) {
        TransmissionTemplate template = buildTemplate(pureContent);

        String taskId = getTaskId(template);
        IPushResult iPushResult = push.pushMessageToList(taskId, targetList(clientIds));

        handleResult(iPushResult);
    }

    /**
     * 批量推送
     *
     * @param pureContents
     * @return
     */
    public List<PureContent> batchPush(List<PureContent> pureContents) {
        List<PureContent> failContent = new ArrayList<>();
        IBatch batch = push.getBatch();
        for (PureContent pureContent : pureContents) {
            if (pureContent.retry() >= MAX_RETRY_TIMES) {
                continue;
            }
            TransmissionTemplate template = buildTemplate(pureContent);
            SingleMessage message = new SingleMessage();
            message.setData(template);
            message.setOffline(true);
            //离线有效时间，单位为毫秒，可选
            message.setOfflineExpireTime(TimeUnit.DAYS.toMillis(1));

            Target target = new Target();
            target.setAppId(appId);
            target.setClientId(pureContent.getClientId());
            try {
                batch.add(message, target);
            } catch (Exception e) {
                failContent.add(pureContent);
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("添加批量推送失败:{}", JSON.toJSONString(pureContent), e);
                }
            }
        }
        return handlerResult(pureContents, failContent, batch);
    }


    /**
     * 全平台的设备推送通知
     *
     * @param pureContent
     */
    public void pushToAll(PureContent pureContent) {

        TransmissionTemplate template = buildTemplate(pureContent);
        AppMessage message = new AppMessage();
        message.setData(template);
        message.setOffline(true);
        //离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(TimeUnit.DAYS.toMillis(1));
        //推送给App的目标用户需要满足的条件
        AppConditions cdt = new AppConditions();
        message.setAppIdList(Collections.singletonList(appId));
        message.setConditions(cdt);

        IPushResult iPushResult = push.pushMessageToApp(message, "");
        handleResult(iPushResult);
    }

    private void handleResult(IPushResult iPushResult) {

        if (null != iPushResult) {
            String result = iPushResult.getResponse().get(GETUI_RESULT).toString();
            if (!GETUI_RESULT_CODE_CID.equals(result) && !"Success".equals(result)) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("请求个推pushMessageToList接口失败，返回参数：{}", JSON.toJSONString(iPushResult));
            }
                throw new IllegalStateException("请求个推接口失败,原因根据日志查询");
            }
        }
    }


    private String getTaskId(TransmissionTemplate template) {
        ListMessage message = new ListMessage();
        message.setData(template);
        // 设置消息离线，并设置离线时间
        message.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(TimeUnit.DAYS.toMillis(1));

        return push.getContentId(message);
    }

    private List<Target> targetList(Collection<String> clients) {
        return clients.stream().distinct().map(cid -> {
            Target target = new Target();
            target.setAppId(appId);
            target.setClientId(cid);
            return target;
        }).collect(Collectors.toList());
    }

    /**
     * 构建通知模版
     * 因为getui离线厂商通道的原因，建议用透传模版加Intent的方式推送通知，所以这边构建的是透传模版
     *
     * @return
     */
    private TransmissionTemplate buildTemplate(PureContent pureContent) {
        String title = pureContent.getTitle();
        String content = pureContent.getDesc();
        String path = pureContent.getAndroidJump();
        Map<String, String> extra = pureContent.getExtra();
        if (extra == null) {
            extra = new HashMap<>(1);
        }
        extra.put("androidJump", path);
        TransmissionTemplate template = new TransmissionTemplate();
        // 设置APPID与APPKEY
        template.setAppId(appId);
        template.setAppkey(appKey);

        PushType pushType = PushType.of(pureContent.getType());
        if (pushType != PushType.THROUGH_DATA) {
            //此处为厂商通道的参数，只是透传时不需要
            Notify notify = new Notify();
            notify.setTitle(title);
            notify.setContent(content);
            notify.setType(GtReq.NotifyInfo.Type._intent);
            StringBuilder intent = new StringBuilder(String.format("intent:#Intent;action=android.intent.action" +
                    ".oppopush;package=%s;component=%s;", packageName, component));
            //构建path
            try {

                String urlPath = "S.path=" + URLEncoder.encode(path, GlobalConstant.DEFAULT_ENCODING) + ";";
                //构建params
                String params = "S.params=" + URLEncoder.encode(JSON.toJSONString(extra),
                        GlobalConstant.DEFAULT_ENCODING) + ";";
                //构建title
                String urlTitle = "S.title=" + URLEncoder.encode(title, GlobalConstant.DEFAULT_ENCODING) + ";";
                //构建conetnt
                String urlContent = "S.content=" + URLEncoder.encode(content, GlobalConstant.DEFAULT_ENCODING) + ";";

                intent.append(urlPath).append(params).append(urlTitle).append(urlContent).append("end");
                notify.setIntent(intent.toString());
                template.set3rdNotifyInfo(notify);
            } catch (Exception e) {
                LOGGER.warn("intent为：{}", intent, e);
            }
        }
        // 透传消息设置，1为强制启动应用，客户端接收到消息后就会立即启动应用；2为等待应用启动
        template.setTransmissionType(2);

        //透传+通知的情况，如果应用离线，则通过厂商通道唤醒并进行通知；如果应用在线，要区分是否在前台
        //理想状态是客户端自行判断应用的使用状况并选择弹窗类型，如果无法实现则只能两个都弹
        extra.put("pushType", pushType.getCode());
        extra.put("title", title);
        extra.put("content", content);
        template.setTransmissionContent(JSON.toJSONString(extra));

        return template;
    }

    private List<PureContent> handlerResult(List<PureContent> pureContents, List<PureContent> failContent,
                                            IBatch batch) {
        try {
            IPushResult result = batch.submit();
            Object object = result.getResponse().get("info");
            if (object != null) {
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(result.getResponse().get("info")));
                for (Map.Entry<String, Object> entry : jsonObject.entrySet()) {
                    String index = entry.getKey();
                    Object value = entry.getValue();
                    if (value instanceof JSONObject) {
                        JSONObject item = (JSONObject) value;
                        if (!GETUI_RESULT_CODE_CID.equals(item.getString(GETUI_RESULT))) {
                            //不成功
                            LOGGER.error("推送失败，cid:{},失败原因:{}", item.getString("cid"), item.getString(GETUI_RESULT));
                            failContent.add(pureContents.get(Integer.parseInt(index) - 1));
                        }
                    }
                }
            }
            return failContent;
        } catch (IOException e) {
            LOGGER.error("推送失败", e);
            return pureContents;
        }
    }
}
