package cn.tuia.explore.center.util;

import cn.tuia.explore.center.constant.GlobalConstant;
import cn.tuia.explore.center.enums.RedisKey;
import com.google.common.base.Joiner;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * redis key组装工具类
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: RedisKeyUtil.java , v 0.1 2017/5/19 下午4:56 ZhouFeng Exp $
 */
public class RedisKeyUtil {

    private static final String SEPARATOR = "_";

    private static final String EC_PREFIX = "EC";

    /**
     * 每日必抢中奖码生成锁
     *
     * @return
     */
    public static String dailyLuckyGenLock() {
        return getKey(RedisKey.K47);
    }


    /**
     * 每日任务的进度
     *
     * @param userId
     * @param missionId
     * @return
     */
    public static String dailyMission(long userId, int missionId) {
        return getKey(RedisKey.K45, userId, missionId, LocalDate.now());
    }
    /**
     * 一元提现每日限制
     *
     * @param userId
     * @return
     */
    public static String withdrawLimit(long userId) {
        return getKey(RedisKey.K44, userId, LocalDate.now());
    }

    /**
     * 加入阅读任务分布式锁
     *
     * @param userId
     * @return
     */
    public static String joinReadTaskLock(long userId) {
        return getKey(RedisKey.K43, userId);
    }

    /**
     * 记录今日是否创建过阅读任务
     *
     * @return
     */
    public static String isCreateReadTask() {
        return getKey(RedisKey.K40, LocalDate.now());
    }

    /**
     * 用户观看视频开始时间
     *
     * @param userId
     * @param postId
     * @return
     */
    public static String startWatch(long userId, long postId) {
        return getKey(RedisKey.K39, userId, postId);
    }

    /**
     * 老用户内容引导消息
     *
     * @param userId
     * @return
     */
    public static String recommendMessageRecord(long userId) {
        return getKey(RedisKey.K38, userId, LocalDate.now());
    }

    /**
     * 用户的开始/结束阅读记录
     *
     * @param uuid
     * @param userId
     * @param type
     * @return
     */
    public static String readRecord(String uuid, long userId, int type) {
        return getKey(RedisKey.K37, uuid, userId, type);
    }

    /**
     * 用户上一次点击该标签动态的时间
     *
     * @param userId
     * @param tag
     * @return
     */
    public static String lastClickTag(long userId, int tag) {
        return getKey(RedisKey.K33, userId, tag);
    }

    /**
     * 用户未读消息提醒
     *
     * @param userId
     * @return
     */
    public static String unreadRedmin(long userId) {
        return getKey(RedisKey.K32, userId);
    }

    /**
     * token与userid的映射
     *
     * @param token
     * @return
     */
    public static String tokenExchange(String token) {
        return getKey(RedisKey.K01, token);
    }

    /**
     * 消息推送分布式锁
     *
     * @return
     */
    public static String pushLockKey() {
        return getKey(RedisKey.K02);
    }

    /**
     * 待推送消息
     *
     * @param time
     * @return
     */
    public static String pushListKey(LocalDateTime time) {
        return getKey(RedisKey.K03, time.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }


    /**
     * 某条记录的点赞数
     */
    public static String likeTimes(long msgId, int msgType) {
        return getKey(RedisKey.K04, msgId, msgType);
    }

    /**
     * 某篇信息的评论数
     */
    public static String commentTimes(long msgId, int msgType) {
        return getKey(RedisKey.K05, msgId, msgType);
    }

    /**
     * 某篇帖子的分享数
     */
    public static String shareTimes(long postId) {
        return getKey(RedisKey.K06, postId);
    }

    /**
     * 待同步点赞数的记录
     */
    public static String waitSyncLike(LocalDateTime time) {
        return getKey(RedisKey.K07, time.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }

    /**
     * 待同步评论数的记录
     */
    public static String waitSyncComment(LocalDateTime time) {
        return getKey(RedisKey.K08, time.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }

    /**
     * 待同步分享数的记录
     */
    public static String waitSyncShare(LocalDateTime time) {
        return getKey(RedisKey.K09, time.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }

    /**
     * 业务计数器
     *
     * @param clazz
     * @param key
     * @return
     */
    public static String timerKey(Class<?> clazz, String key) {
        return getKey(RedisKey.K10, clazz.getSimpleName(), key);
    }

    /**
     * 待同步的业务计数器
     *
     * @param clazz
     * @param time
     * @return
     */
    public static String counterWaitSync(Class<?> clazz, LocalDateTime time) {
        return getKey(RedisKey.K11, clazz.getSimpleName(), time.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }

    /**
     * 访问主页分布式锁
     *
     * @param vistorUserId
     * @param masterUserId
     * @return
     */
    public static String visitLock(long vistorUserId, long masterUserId) {
        return getKey(RedisKey.K12, vistorUserId, masterUserId);
    }

    /**
     * 子评论列表
     *
     * @param commentId
     * @return
     */
    public static String subCommentList(long commentId) {
        return getKey(RedisKey.K13, commentId);
    }

    /**
     * 待同步的子评论
     *
     * @return
     */
    public static String waitSyncSubComment(LocalDateTime localDateTime) {
        return getKey(RedisKey.K14,localDateTime.format(DateTimeFormatter.ofPattern("yyyyMMddHHmm")));
    }

    /**
     * 用户推荐列表排序缓存
     *
     * @return
     */
    public static String userFollowRcmd(Date date) {
        return getKey(RedisKey.K15, new SimpleDateFormat("yyyyMMdd").format(date));
    }

    /**
     * 手机验证码
     *
     * @param phoneNum
     * @return
     */
    public static String smsVerify(String phoneNum) {
        return getKey(RedisKey.K17, phoneNum);
    }

    /**
     * 更新record表时的分布式锁
     *
     * @param userKey
     * @return
     */
    public static String recordLockKey(String userKey) {
        return getKey(RedisKey.K19, userKey);
    }

    /**
     * 动态的实时得分
     * @param postId
     * @return
     */
    public static String postScore(long postId) {
        return getKey(RedisKey.K20, postId);
    }

    /**
     * 动态曝光时记录，用来与后续点击进行匹配
     *
     * @param deviceId
     * @param postId
     * @return
     */
    public static String postExposure(String deviceId, long postId) {
        return getKey(RedisKey.K21, deviceId, postId);
    }

    /**
     * 进入动态时记录，用户与后续退出动态进行匹配
     *
     * @param deviceId
     * @param postId
     * @return
     */
    public static String postEnter(String deviceId, long postId) {
        return getKey(RedisKey.K22, deviceId, postId);
    }

    /**
     * 退出动态时记录，用来防止多次退出计数
     *
     * @param deviceId
     * @param postId
     * @return
     */
    public static String postExit(String deviceId, long postId) {
        return getKey(RedisKey.K23, deviceId, postId);
    }

    /**
     * 待推荐的内容id列表缓存
     *
     * @param size
     * @param currentPage
     * @param type
     * @return
     */
    public static String postPoolCache(int size, int currentPage, int type) {
        return getKey(RedisKey.K24, size, currentPage, type);
    }

    /**
     * 得分清零分布式锁
     *
     * @return
     */
    public static String postScoreClearKey(String deviceId, long postId) {
        return getKey(RedisKey.K25, deviceId, postId);
    }

    /**
     * 每秒钟待推送的数据
     *
     * @param localDateTime
     * @return
     */
    public static String waitPushList(LocalDateTime localDateTime) {
        return getKey(RedisKey.K26, localDateTime.format(GlobalConstant.DEFAULT_DATE_TIME_FOMATTER));
    }

    /**
     * 内容的曝光数
     *
     * @param id
     * @param date
     * @return
     */
    public static String exposureKey(String id, String date) {
        return getKey(RedisKey.K27, id, date);
    }

    /**
     * 内容的点击数
     *
     * @param id
     * @param date
     * @return
     */
    public static String clickKey(String id, String date) {
        return getKey(RedisKey.K28, id, date);
    }

    /**
     * 频道内容id列表缓存
     *
     * @param size
     * @param currentPage
     * @param type
     * @return
     */
    public static String postChannelCache(int size, int currentPage, int type) {
        return getKey(RedisKey.K30, size, currentPage, type);
    }


    private static String getKey(RedisKey redisKey, Object... objects) {
        return getKey(EC_PREFIX, redisKey, objects);
    }

    private static String getKey(String prefix, RedisKey redisKey, Object... objects) {


        Joiner joiner = Joiner.on(SEPARATOR).skipNulls();
        List<Object> items = new ArrayList<>();
        items.add(prefix);
        items.add(redisKey);
        items.addAll(Arrays.stream(objects).map(value -> value == null ? "null" : value).collect(Collectors.toList()));
        return joiner.join(items);

    }

    private RedisKeyUtil() {
        // 不允许创建实例
    }


}
