package cn.tuia.payment.api.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Objects;

/**
 * @author y.zhao
 * @title: DateUtils
 * @projectName tuia-payment-center
 * @description: description
 */
public class DateUtils {
    private static final String DATE_FORMAT_DEFAULT = "yyyy-MM-dd HH:mm:ss";

    /**
     * 将符合特定格式的日期字符串转换为Date对象。
     *
     * @param dateString 符合"yyyy-MM-dd HH:mm:ss"格式的日期字符串
     * @return 解析后的Date对象，如果字符串不符合格式则返回null
     */
    public static Date stringToDate(String dateString) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT_DEFAULT);
        try {
            return dateFormat.parse(dateString);
        } catch (ParseException e) {
            // 如果字符串不符合格式，打印错误并返回null或抛出自定义异常
            // System.err.println("Error parsing date: " + e.getMessage());
            return null;
        }
    }

    /**
     * 获取当前时间 yyyy-MM-dd HH:mm:ss 格式
     *
     * @return
     */
    public static String getDataNow() {
        long timestamp = System.currentTimeMillis();
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT_DEFAULT, Locale.ENGLISH);
        return sdf.format(new Date(timestamp));
    }

    // 获取今天的 00:00:00，返回 Date 类型
    public static Date getTodayStart() {
        LocalDateTime todayStart = LocalDateTime.of(LocalDateTime.now().toLocalDate(), LocalTime.MIDNIGHT);
        return Date.from(todayStart.atZone(ZoneId.systemDefault()).toInstant());
    }

    // 获取上一个小时的时间段范围，返回 Date[] 类型
    public static Date[] getPreviousHourRange() {
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime startOfCurrentHour = now.withMinute(0).withSecond(0).withNano(0);
        LocalDateTime startOfPreviousHour = startOfCurrentHour.minusHours(1);

        Date start = Date.from(startOfPreviousHour.atZone(ZoneId.systemDefault()).toInstant());
        Date end = Date.from(startOfCurrentHour.atZone(ZoneId.systemDefault()).toInstant());

        return new Date[]{start, end};
    }

    /**
     * 获取当前时间 yyyyMMdd 格式
     *
     * @return
     */
    public static String getData() {
        return new SimpleDateFormat("yyyyMMdd").format(new Date());
    }

    public static Date getTomorrowStartTime(Integer day) {
        // 创建一个日历实例
        Calendar calendar = Calendar.getInstance();
        // 将日历日期设置为下一天的00:00:00
        calendar.add(Calendar.DATE, Objects.isNull(day) ? 1 : day);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        // 获取对应的Date对象
        return calendar.getTime();
    }


    public static void main(String[] args) {
        // 获取今天的 00:00:00
        Date todayStart = getTodayStart();
        System.out.println("今天的 00:00:00: " + todayStart);

        // 获取上一个小时时间段范围
        Date[] previousHourRange = getPreviousHourRange();
        System.out.println("上一个小时时间范围: " + previousHourRange[0] + " 到 " + previousHourRange[1]);
    }
}
