package com.qiho.center.api.enums.ShotOrder;

import com.qiho.center.api.constant.ChuangLanConstant;
import com.qiho.center.api.constant.IDCardConstant;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.area.AreaCodeEnum;
import com.qiho.center.api.enums.bean.LocationCheckBean;
import com.qiho.center.api.util.AddressUtil;
import com.qiho.center.api.util.IdCardUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Created by danke on 2017/6/21.
 * 定义校验规则及对应的校验方法
 */
@Slf4j
public enum ShotOrderRuleEnum implements ShotOrderValute {

    // 说明：threshold是策略配置的阈值，hopeVal是当前订单产生的数值

    LESS_THAN(0,"小于","lesser"){
        /**
         * threshold为阈值,hopeVal为期望值属性值,当threshold > hopeVal时,返回ture,命中则返回ture
         * @param threshold
         * @param hopeVal
         * @return
         */
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) > Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }

    },
    LESS_EQUALS(1,"小于等于","lesserEquals"){
        /**
         * threshold为阈值,hopeVal为期望值属性值,当a > b时,返回ture,命中则返回ture
         * @param threshold
         * @param hopeVal
         * @return
         */
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) >= Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }

    },
    GREATER_EQUALS(2,"大于等于","greaterEquals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) <= Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }
    },
    GREATER_THAN(3,"大于","greater"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) < Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }
    },

    EQUALS_THAN(4,"等于","equals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            return ResultBase.rightReturn(threshold.equals(String.valueOf(hopeVal)));
        }
    },

    CONTAINS_THAN(5,"包含","contains"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (null == threshold || null == hopeVal)
                return ResultBase.rightReturn(Boolean.FALSE);
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = threshold.split(",");
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount > 0);
        }
    },

    EMPTY_THAN(6,"为空","hollow"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(hopeVal == null ? Boolean.TRUE : StringUtils.isBlank(String.valueOf(hopeVal)));
        }
    },

    NOT_EQUALS_THAN(6,"不等于","notequals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            return ResultBase.rightReturn(!threshold.equals(String.valueOf(hopeVal)));
        }
    },
    NOT_EMPTY_THAN(7,"不为空","nothollow"){
		@Override
		public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal){
			return ResultBase.rightReturn(hopeVal == null ? Boolean.FALSE: StringUtils.isNotBlank(String.valueOf(hopeVal)));
		}
    },
    //包含中文逗号和英文逗号两种
    CONTAINS_COMMA(8,"包含逗号","containsComma"){
		@Override
		public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
			if (null == hopeVal){
				 return ResultBase.rightReturn(Boolean.FALSE);
			}
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = new String[]{",","，"};
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount > 0);
		}
    },

    NOT_CONTAINS_THAN(9,"不包含","notcontains"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (null == threshold || null == hopeVal)
                return ResultBase.rightReturn(Boolean.FALSE);
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = threshold.split(",");
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount.intValue() == 0 );
        }
    },

    /**
     * 创蓝 手机号状态: 未知
     */
    UN_KNOWS(10, "未知", "unknowns"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.UNKNOWN, hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：空号
     */
    EMPTY_NUM(11, "空号", "emptyNum"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.EMPTY_NUM,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：超时
     */
    TIME_OUT(12, "超时", "timeOut"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.TIME_OUT,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：关机或停机
     */
    CLOSE_DOWN(13, "关机或停机", "closeDown"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.CLOSE_DOWN,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：黑名单
     */
    BLACKLIST(14, "黑名单", "blacklist"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.BLACKLIST,hopeValStr));
        }
    },

    /**
     * 完全不同（省）
     */
    ALL_DIFFER(15, "完全不同（省）", "ALL_DIFFER"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getProvince().equals(bean.getIpProvince())&&!bean.getProvince().equals(bean.getMobileProvince())
                &&!bean.getIpProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 手机号归属地和收货地址不同（到省）
     */
    MOBILE_ADDRESS_DIFFER(16, "手机号归属地和收货地址不同（到省）", "MOBILE_ADDRESS_DIFFER"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 手机号归属地和收货地址不同（到省市）
     */
    MOBILE_ADDRESS_DIFFER_CITY(17, "手机号归属地和收货地址不同（到省市）", "MOBILE_ADDRESS_DIFFER_CITY"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getCity().equals(bean.getMobileCity())||!bean.getProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 详细地址不包含中文
     */
    NOT_CONTAIN_CN(12, "不包含中文", "notContainCN"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(notContainCN(hopeValStr));
        }

        private  boolean notContainCN(String str) {

            //中文regex
            String regex = "[\\u4e00-\\u9fa5]";

            Pattern p = Pattern.compile(regex);
            Matcher m = p.matcher(str);
            if (m.find()) {
                return false;
            }

            return true;
        }
    },

    ADDRESS_INVALID(13, "地址无效", "addressInvalid"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            // 高德地图校验详情地址是否与所选市对应，hopeVal为根据地址查询到的城市数量
            return ResultBase.rightReturn(Integer.valueOf(String.valueOf(hopeVal)) <= 0);
        }
    },

    /**
     * 这里的包含的非 {@link ShotOrderRuleEnum#CONTAINS_THAN} 只是文案一样 但计算规则不同 这里是用来计算收货地址包含于指定的范围
     *
     * 方法中用到的符号 均是和前端约定好 省-城市&城市,省-城市&城市
     */
    ADDRESS_OUT_OF_RANGE(14,"包含","addressOutOfRange"){
        @Override
        public ResultBase<Boolean>  valuteOrderByQuery(String threshold, Object hopeVal) {

            //如果没有选择指定的城市 则默认放行
            if(org.apache.commons.lang3.StringUtils.isEmpty(threshold)){
                return ResultBase.rightReturn(Boolean.FALSE);
            }

            LocationCheckBean bean = (LocationCheckBean)hopeVal;

            // 下单省份，截取前两字符，因为省市自治区至少两个字，但由于这个填写表单地址，广告主上传的省市地址字符无法预测，万一只有一个字符
            String orderProvince = bean.getProvince();
            if (orderProvince.length() >= 2) {
                orderProvince = orderProvince.substring(0, 2);
            }

            // 下单城市，同上
            String orderCity = bean.getCity();
            if (orderCity.length() >= 2) {
                orderCity = orderCity.substring(0, 2);
            }

            //如果城市和省份为空 则不进行校验 已和产品同步 @张珊
            if(StringUtils.isEmpty(orderProvince) || StringUtils.isEmpty(orderCity)){
                return ResultBase.rightReturn(false);
            }

            //key-> 省份 value -> 城市
            Map<String, Set<String>> provinceCityMap = new HashMap<>(16);

            String[] provinceAndCities = threshold.split(",");
            for (String provinceCity : provinceAndCities) {
                if(StringUtils.isEmpty(provinceCity)){
                    continue;
                }
                String[] provinceCityArray = provinceCity.split("-");

                // 校验省份，截取前面两个字符，因为有些广告主上传的表单地址库可能是北京 宁夏等，而防作弊存储的是北京市 宁夏回族自治区
                String province = provinceCityArray[0].substring(0, 2);

                // 规则城市同理取前两字符,因为防作弊地区中的省市都是三个字及以上,所以substring(0,2)不会有StringIndexOutOfBoundsException
                Set<String> cities = provinceCityArray.length > 1 ? Arrays.stream(provinceCityArray[1].split("&"))
                        .map(city -> city.substring(0, 2)).collect(Collectors.toSet()) : new HashSet<>();

                provinceCityMap.put(province, cities);
            }

            // 直辖市只判断省级
            if (AddressUtil.isMunicipality(orderProvince) && provinceCityMap.containsKey(orderProvince)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }

            Set<String> cityList = provinceCityMap.get(orderProvince);

            if(cityList == null || !cityList.contains(orderCity)){
                return ResultBase.rightReturn(Boolean.FALSE);
            }

            return ResultBase.rightReturn(Boolean.TRUE);
        }
    },

    ID_CARD_AGE_INVALID(18, "身份证年龄无效", "idCardAgeInvalid"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (Objects.isNull(hopeVal)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            String idCard = String.valueOf(hopeVal);
            if (idCard.matches(IDCardConstant.REGEX_ID_CARD_15)) {
                return ResultBase.rightReturn(Boolean.FALSE); // 15位身份证不校验年龄
            }
            if (!idCard.matches(IDCardConstant.REGEX_ID_CARD_18)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");

            try {
                Date birthDate = sdf.parse(idCard.substring(6, 14));
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(new Date());
                calendar.add(Calendar.YEAR, -18);
                calendar.getTime();
                return ResultBase.rightReturn(birthDate.compareTo(calendar.getTime()) > 0);
            } catch (ParseException e1) {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    ID_CARD_FORMAT_INVALID(19, "身份证格式无效", "idCardFormatInvalid"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (Objects.isNull(hopeVal)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            String idCard = String.valueOf(hopeVal);
            // 新增除了身份证号长度相关的规则
            if (idCard.matches(IDCardConstant.REGEX_ID_CARD_18) && IdCardUtil.checkIdCardFormat(idCard)) {
                return ResultBase.rightReturn(Boolean.FALSE);
            }

            return ResultBase.rightReturn(Boolean.TRUE);
        }
    },

    ID_CARD_AREA_INVALID(20, "身份证区号无效", "idCardAreaInvalid"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (Objects.isNull(hopeVal)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            String idCard = String.valueOf(hopeVal);
            if (idCard.matches(IDCardConstant.REGEX_ID_CARD_15)) {
                return ResultBase.rightReturn(Boolean.FALSE); // 15位身份证不校验区号
            }

            if (!idCard.matches(IDCardConstant.REGEX_ID_CARD_18)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            AreaCodeEnum areaCodeEnum = AreaCodeEnum.getByCode(idCard.substring(0,2));
            if (Objects.isNull(areaCodeEnum)){
                return ResultBase.rightReturn(Boolean.TRUE);
            }

            return ResultBase.rightReturn(!areaCodeEnum.getAreaCodeEnumerable().isExist(idCard.substring(0,6)));
        }
    },

    ID_CARD_AGE_GREATER_THAN(21, "大于", "idCardAgeGreaterThan"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (Objects.isNull(hopeVal)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            String idCard = String.valueOf(hopeVal);
            if (idCard.matches(IDCardConstant.REGEX_ID_CARD_15)) {
                return ResultBase.rightReturn(Boolean.FALSE); // 15位身份证不校验年龄
            }
            if (!idCard.matches(IDCardConstant.REGEX_ID_CARD_18)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");

            try {
                Date birthDate = sdf.parse(idCard.substring(6, 14));
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(new Date());
                calendar.add(Calendar.YEAR, Integer.valueOf(threshold) * -1);
                calendar.getTime();
                return ResultBase.rightReturn(birthDate.compareTo(calendar.getTime()) < 0);
            } catch (ParseException e1) {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    ID_CARD_AGE_LESS_THAN(22, "小于", "idCardAgeLessThan"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (Objects.isNull(hopeVal)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            String idCard = String.valueOf(hopeVal);
            if (idCard.matches(IDCardConstant.REGEX_ID_CARD_15)) {
                return ResultBase.rightReturn(Boolean.FALSE); // 15位身份证不校验年龄
            }
            if (!idCard.matches(IDCardConstant.REGEX_ID_CARD_18)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");

            try {
                Date birthDate = sdf.parse(idCard.substring(6, 14));
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(new Date());
                calendar.add(Calendar.YEAR, Integer.valueOf(threshold) * -1);
                calendar.getTime();
                return ResultBase.rightReturn(birthDate.compareTo(calendar.getTime()) > 0);
            } catch (ParseException e1) {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    NOT_ONLY_CN(23, "非纯中文", "notOnlyCN"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            String hopeValStr = String.valueOf(hopeVal);
            return ResultBase.rightReturn(!isFullChinese(hopeValStr));
        }

        private boolean isFullChinese(String str) {
            String reg = "[\\u4e00-\\u9fa5]+";
            return str.matches(reg);
        }
    },
    ID_NAME_INVALID(24, "身份证名字不匹配", "idNameInvalid"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(true);
        }
    },
    MOBILE_QUALIFICATION_INVALID(25, "手机号无效", "mobileQualificationInvalid"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(true);
        }
    },

    ZUOYEBANG_OVERSOLD(26, "超出作业帮商品库存", "zuoyebangOversold"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(true);
        }
    },

    CUSTOM_CONFIG_ERROR(27, "自定义配置请求异常", "customConfigError"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(true);
        }
    },

    HIT_UNICOM_CARD_RISK_CONTROL(27, "风控校验不通过", "hitUnicomCardRiskControl"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(true);
        }
    },

    /**
     * 用户填写的详细地址和省市区按照一定的规则命中则认为订单无效
     */
    ADDRESS_REGION_REPEAT(28, "重复填写地区", "addressRegionRepeat") {
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            String[] province_city_distract = threshold.split("&");
            String address = String.valueOf(hopeVal).replaceAll("[省|市|区]", "");
            // 组装匹配库，省、市、区、省&市、省&区、市&区、省&市&区，一共七种组合（不交换顺序）。
            Set<String> mapping = new HashSet<>();
            mapping.add(province_city_distract[0]);
            mapping.add(province_city_distract[1]);
            mapping.add(province_city_distract[2]);
            mapping.add(province_city_distract[0] + province_city_distract[1]);
            mapping.add(province_city_distract[0] + province_city_distract[2]);
            mapping.add(province_city_distract[1] + province_city_distract[2]);
            mapping.add(province_city_distract[0] + province_city_distract[1] + province_city_distract[2]);
            if (mapping.contains(address)) {
                return ResultBase.rightReturn(true);
            }
            return ResultBase.rightReturn(false);
        }
    }
    ;

    int code;//状态码
    String msg;//前端页面显示信息
    String val;//规则修饰,落库

    /**
     * @param code
     * @return
     */
    public static ShotOrderRuleEnum fromStatus(Integer code) {
        if (code==null) {
            return null;
        }
        for (ShotOrderRuleEnum shotOrderRuleEnum : ShotOrderRuleEnum.values()) {
            if (shotOrderRuleEnum.getCode() == code) {
                return shotOrderRuleEnum;
            }
        }
        return null;
    }

    /**
     * 通过val值去获取对应的枚举
     * @param val
     */
    public static ShotOrderRuleEnum fromVal(String val){
        if(StringUtils.isBlank(val)){
            return null;
        }
        for (ShotOrderRuleEnum shotOrderRuleEnum : ShotOrderRuleEnum.values()) {
            if (shotOrderRuleEnum.getVal().equals(val)) {
                return shotOrderRuleEnum;
            }
        }
        return null;
    }

    ShotOrderRuleEnum(int code, String msg,String val) {
        this.code = code;
        this.msg = msg;
        this.val = val;
    }

    public int getCode() {
        return code;
    }

    public String getMsg() {
        return msg;
    }

    public String getVal() {
        return val;
    }



}
