package com.qiho.center.api.util;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.order.DeliveryRecordDto;
import com.qiho.center.api.params.OrderFileDeliveryParam;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * FileDeliveryUtil
 *
 * @Created by zhangshun on 2018-03-10 15:42:28
 */
public class FileDeliveryUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileDeliveryUtil.class);

    private FileDeliveryUtil() {

    }

    public synchronized static int getExcelRowCount(String ossUrl, InputStream input) {
        Assert.notNull(ossUrl, "Excel文件名为空，ossUrl：" + ossUrl);
        Assert.notNull(input, "Excel文件流为空，ossUrl：" + ossUrl);

        //获得当前sheet工作表
        Sheet sheet = FileDeliveryUtil.getSheet(ossUrl, input);
        Assert.notNull(sheet, "Excel文件Sheet为空，ossUrl：" + ossUrl);

        return sheet.getLastRowNum();
    }

    public synchronized static List<OrderFileDeliveryParam> getExcelContent(DeliveryRecordDto recordDto,
        InputStream input) {
        Assert.notNull(recordDto, "参数为空");
        Assert.notNull(recordDto.getFileUrl(), "OSS文件名为空");
        Assert.notNull(recordDto.getId(), "任务ID，ossUrl：" + recordDto.getFileUrl());
        Assert.notNull(recordDto.getMerchantId(), "商家ID为空，ossUrl：" + recordDto.getFileUrl());
        Assert.notNull(recordDto.getLogisticsCode(), "物流编码为空，ossUrl：" + recordDto.getFileUrl());
        Assert.notNull(input, "Excel文件流为空，ossUrl：" + recordDto.getFileUrl());

        //获得当前sheet工作表
        Sheet sheet = FileDeliveryUtil.getSheet(recordDto.getFileUrl(), input);
        Assert.notNull(sheet, "Excel文件Sheet为空，ossUrl：" + recordDto.getFileUrl());

        //获得当前sheet的开始行
        int firstRowNum = sheet.getFirstRowNum();

        //获得当前sheet的结束行
        int lastRowNum = sheet.getLastRowNum();

        DeliveryHeaderNumber header = getHeaderNumber(sheet.getRow(0));

        OrderFileDeliveryParam bean = null;
        List<OrderFileDeliveryParam> list = Lists.newArrayList();

        //循环除了第一行的所有行
        for (int rowNum = firstRowNum + 1; rowNum <= lastRowNum; rowNum++) {
            try {
                //获得当前行
                bean = setCellValue(rowNum + 1, header, sheet.getRow(rowNum));
                if (null != bean) {
                    list.add(bean);
                }
            } catch (Exception e) {
                LOGGER.warn("解析文件错误，ossUrl：" + recordDto.getFileUrl());
            }
        }

        return list;
    }

    private static OrderFileDeliveryParam setCellValue(int rowNum, DeliveryHeaderNumber header, Row row) {
        if (row == null) {
            return null;
        }

        OrderFileDeliveryParam bean = new OrderFileDeliveryParam();
        bean.setRowNumber(rowNum);

        Cell cellErp = null, cellPost = null;

        if (null != header.getErp()) {
            cellErp = row.getCell(header.getErp());
        }

        if (null != header.getPost()) {
            cellPost = row.getCell(header.getPost());
        }

        if (null == cellErp) {
            bean.setErpId("");
        } else {
            bean.setErpId(getCellValue(cellErp));
        }

        if (null == cellPost) {
            bean.setPostId("");
        } else {
            bean.setPostId(getCellValue(cellPost));
        }

        if (StringUtils.isBlank(bean.getErpId()) && StringUtils.isBlank(bean.getPostId())) {
            return null;
        }

        return bean;
    }

    public static DeliveryHeaderNumber getHeaderNumber(Row row) {

        if (null == row) {
            return new DeliveryHeaderNumber(0, 1);
        }

        //获得当前行的开始列
        int first = row.getFirstCellNum();

        //获得当前行的列数
        int last = row.getLastCellNum();

        DeliveryHeaderNumber cellNumber = new DeliveryHeaderNumber();

        Cell cell = null;
        for (int cellNum = first; cellNum < last; cellNum++) {

            cell = row.getCell(cellNum);

            if (null == cell) {
                continue;
            }

            if ("待发货订单ID".equals(getCellValue(cell)) || "待发货订单id".equals(getCellValue(cell)) || "待发货订单ｉｄ"
                .equals(getCellValue(cell))) {
                cellNumber.setErp(cellNum);
            }
            if ("物流单号".equals(getCellValue(cell))) {
                cellNumber.setPost(cellNum);
            }
        }

        return cellNumber;
    }

    public static Sheet getSheet(String ossUrl, InputStream input) {
        //创建Workbook工作薄对象，表示整个excel
        Sheet sheet = null;
        Workbook workbook = null;
        try {

            //根据文件后缀名不同(xls和xlsx)获得不同的Workbook实现类对象
            if (ossUrl.endsWith("xls")) {
                //2003
                workbook = new HSSFWorkbook(input);
            } else if (ossUrl.endsWith("xlsx")) {
                //2007
                workbook = new XSSFWorkbook(input);
            }

            if (null == workbook) {
                return null;
            }

            sheet = workbook.getSheetAt(0);

            if (null == sheet) {
                return null;
            }

            return sheet;

        } catch (IOException e) {
            LOGGER.error("获取WorkBook错误，Url：{}", ossUrl);
            return null;
        }
    }

    public static String getCellValue(Cell cell) {
        if (null == cell) {
            return StringUtils.EMPTY;
        }
        try {

            if (HSSFCell.CELL_TYPE_NUMERIC == cell.getCellType()) {
                return String.valueOf(((Double) cell.getNumericCellValue()).longValue());
            }

            return StringUtils.trim(cell.getStringCellValue());

        } catch (Exception e) {
            return StringUtils.EMPTY;
        }
    }

    private static class DeliveryHeaderNumber {

        private Integer erp;

        private Integer post;

        public DeliveryHeaderNumber() {
        }

        public DeliveryHeaderNumber(Integer erp, Integer post) {
            this.erp = erp;
            this.post = post;
        }

        public Integer getErp() {
            return erp;
        }

        public void setErp(Integer erp) {
            this.erp = erp;
        }

        public Integer getPost() {
            return post;
        }

        public void setPost(Integer post) {
            this.post = post;
        }
    }

}
