package com.qiho.center.api.enums.ShotOrder;

import com.qiho.center.api.constant.ChuangLanConstant;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.bean.LocationCheckBean;
import com.qiho.center.api.util.AddressUtil;
import org.apache.commons.lang.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by danke on 2017/6/21.
 * 定义校验规则及对应的校验方法
 */
public enum ShotOrderRuleEnum implements ShotOrderValute {

    // 说明：threshold是策略配置的阈值，hopeVal是当前订单产生的数值

    LESS_THAN(0,"小于","lesser"){
        /**
         * threshold为阈值,hopeVal为期望值属性值,当threshold > hopeVal时,返回ture,命中则返回ture
         * @param threshold
         * @param hopeVal
         * @return
         */
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) > Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }

    },
    LESS_EQUALS(1,"小于等于","lesserEquals"){
        /**
         * threshold为阈值,hopeVal为期望值属性值,当a > b时,返回ture,命中则返回ture
         * @param threshold
         * @param hopeVal
         * @return
         */
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) >= Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }

    },
    GREATER_EQUALS(2,"大于等于","greaterEquals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) <= Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }
    },
    GREATER_THAN(3,"大于","greater"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            Boolean result = Integer.valueOf(threshold) < Integer.valueOf(String.valueOf(hopeVal));
            return ResultBase.rightReturn(result);
        }
    },

    EQUALS_THAN(4,"等于","equals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            return ResultBase.rightReturn(threshold.equals(String.valueOf(hopeVal)));
        }
    },

    CONTAINS_THAN(5,"包含","contains"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (null == threshold || null == hopeVal)
                return ResultBase.rightReturn(Boolean.FALSE);
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = threshold.split(",");
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount > 0);
        }
    },

    EMPTY_THAN(6,"为空","hollow"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            return ResultBase.rightReturn(hopeVal == null ? Boolean.TRUE : StringUtils.isBlank(String.valueOf(hopeVal)));
        }
    },

    NOT_EQUALS_THAN(6,"不等于","notequals"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            ResultBase<Boolean> resultBase = ShotOrderValute.baseParamsCheck(threshold,hopeVal);
            if(!resultBase.hasSuccessValue())
                return resultBase;
            return ResultBase.rightReturn(!threshold.equals(String.valueOf(hopeVal)));
        }
    },
    NOT_EMPTY_THAN(7,"不为空","nothollow"){
		@Override
		public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal){
			return ResultBase.rightReturn(hopeVal == null ? Boolean.FALSE: StringUtils.isNotBlank(String.valueOf(hopeVal)));
		}
    },
    //包含中文逗号和英文逗号两种
    CONTAINS_COMMA(8,"包含逗号","containsComma"){
		@Override
		public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
			if (null == hopeVal){
				 return ResultBase.rightReturn(Boolean.FALSE);
			}
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = new String[]{",","，"};
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount > 0);
		}
    },

    NOT_CONTAINS_THAN(9,"不包含","notcontains"){
        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            if (null == threshold || null == hopeVal)
                return ResultBase.rightReturn(Boolean.FALSE);
            String hopeValStr = String.valueOf(hopeVal);
            String[] thresholdStr = threshold.split(",");
            Long filterCount = Arrays.stream(thresholdStr).filter(e ->StringUtils.contains(hopeValStr,e)).count();
            return ResultBase.rightReturn(filterCount.intValue() == 0 );
        }
    },

    /**
     * 创蓝 手机号状态: 未知
     */
    UN_KNOWS(10, "未知", "unknowns"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.UNKNOWN, hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：空号
     */
    EMPTY_NUM(11, "空号", "emptyNum"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.EMPTY_NUM,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：超时
     */
    TIME_OUT(12, "超时", "timeOut"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.TIME_OUT,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：关机或停机
     */
    CLOSE_DOWN(13, "关机或停机", "closeDown"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.CLOSE_DOWN,hopeValStr));
        }
    },

    /**
     * 创蓝 手机号状态：黑名单
     */
    BLACKLIST(14, "黑名单", "blacklist"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(StringUtils.equals(ChuangLanConstant.BLACKLIST,hopeValStr));
        }
    },

    /**
     * 完全不同（省）
     */
    ALL_DIFFER(15, "完全不同（省）", "ALL_DIFFER"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getProvince().equals(bean.getIpProvince())&&!bean.getProvince().equals(bean.getMobileProvince())
                &&!bean.getIpProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 手机号归属地和收货地址不同（到省）
     */
    MOBILE_ADDRESS_DIFFER(16, "手机号归属地和收货地址不同（到省）", "MOBILE_ADDRESS_DIFFER"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 手机号归属地和收货地址不同（到省市）
     */
    MOBILE_ADDRESS_DIFFER_CITY(17, "手机号归属地和收货地址不同（到省市）", "MOBILE_ADDRESS_DIFFER_CITY"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {
            LocationCheckBean bean = (LocationCheckBean)hopeVal;
            if(!bean.getCity().equals(bean.getMobileCity())||!bean.getProvince().equals(bean.getMobileProvince())){
                return ResultBase.rightReturn(Boolean.TRUE);
            }else {
                return ResultBase.rightReturn(Boolean.FALSE);
            }
        }
    },

    /**
     * 详细地址不包含中文
     */
    NOT_CONTAIN_CN(12, "不包含中文", "notContainCN"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            String hopeValStr = String.valueOf(hopeVal);

            return ResultBase.rightReturn(notContainCN(hopeValStr));
        }

        private  boolean notContainCN(String str) {

            //中文regex
            String regex = "[\\u4e00-\\u9fa5]";

            Pattern p = Pattern.compile(regex);
            Matcher m = p.matcher(str);
            if (m.find()) {
                return false;
            }

            return true;
        }
    },

    ADDRESS_INVALID(13, "地址无效", "addressInvalid"){

        @Override
        public ResultBase<Boolean> valuteOrderByQuery(String threshold, Object hopeVal) {

            // 高德地图校验详情地址是否与所选市对应，hopeVal为根据地址查询到的城市数量
            return ResultBase.rightReturn(Integer.valueOf(String.valueOf(hopeVal)) <= 0);
        }
    },

    /**
     * 这里的包含的非 {@link ShotOrderRuleEnum#CONTAINS_THAN} 只是文案一样 但计算规则不同 这里是用来计算收货地址包含于指定的范围
     *
     * 方法中用到的符号 均是和前端约定好 省-城市&城市,省-城市&城市
     */
    ADDRESS_OUT_OF_RANGE(14,"包含","addressOutOfRange"){
        @Override
        public ResultBase<Boolean>  valuteOrderByQuery(String threshold, Object hopeVal) {

            //如果没有选择指定的城市 则默认放行
            if(org.apache.commons.lang3.StringUtils.isEmpty(threshold)){
                return ResultBase.rightReturn(Boolean.FALSE);
            }

            LocationCheckBean bean = (LocationCheckBean)hopeVal;

            //下单省份
            String orderProvince = bean.getProvince() ;
            //下单城市
            String orderCity = bean.getCity();
            //如果城市和省份为空 则不进行校验 已和产品同步 @张珊
            if(StringUtils.isEmpty(orderProvince) || StringUtils.isEmpty(orderCity)){
                return ResultBase.rightReturn(false);
            }

            //key-> 省份 value -> 城市
            Map<String, Set<String>>  provinceCityMap = new HashMap<>(16);

            for (String provinceCity : threshold.split(",")) {
                if(StringUtils.isEmpty(provinceCity)){
                    continue;
                }
                String[] provinceCityArray = provinceCity.split("-");

                //校验省份
                String province = provinceCityArray[0];

                //规则城市
                Set<String> citys = provinceCityArray.length > 1 ? Optional.ofNullable(provinceCityArray[1]).map(temp -> Stream.of(temp.split("&")).collect(Collectors.toSet())).orElse(new HashSet<>()) : new HashSet<>();
                provinceCityMap.put(province ,citys );
            }

//            if(orderProvince.endsWith("市")){
//                orderProvince=  orderProvince.replace("市", StringUtils.EMPTY);
//
//            }
//            if(orderProvince.endsWith("省")){
//                orderProvince=  orderProvince.replace("省", StringUtils.EMPTY);
//
//            }

            // 直辖市只判断省级
            if (AddressUtil.isMunicipality(orderProvince) && provinceCityMap.containsKey(orderProvince)) {
                return ResultBase.rightReturn(Boolean.TRUE);
            }

            Set<String> cityList = provinceCityMap.get(orderProvince);

            if(cityList == null || !cityList.contains(orderCity)){
                return ResultBase.rightReturn(Boolean.FALSE);
            }

            return ResultBase.rightReturn(Boolean.TRUE);
        }
    }

    ;





    int code;//状态码
    String msg;//前端页面显示信息
    String val;//规则修饰,落库

    /**
     * @param code
     * @return
     */
    public static ShotOrderRuleEnum fromStatus(Integer code) {
        if (code==null) {
            return null;
        }
        for (ShotOrderRuleEnum shotOrderRuleEnum : ShotOrderRuleEnum.values()) {
            if (shotOrderRuleEnum.getCode() == code) {
                return shotOrderRuleEnum;
            }
        }
        return null;
    }

    /**
     * 通过val值去获取对应的枚举
     * @param val
     */
    public static ShotOrderRuleEnum fromVal(String val){
        if(StringUtils.isBlank(val)){
            return null;
        }
        for (ShotOrderRuleEnum shotOrderRuleEnum : ShotOrderRuleEnum.values()) {
            if (shotOrderRuleEnum.getVal().equals(val)) {
                return shotOrderRuleEnum;
            }
        }
        return null;
    }

    ShotOrderRuleEnum(int code, String msg,String val) {
        this.code = code;
        this.msg = msg;
        this.val = val;
    }

    public int getCode() {
        return code;
    }

    public String getMsg() {
        return msg;
    }

    public String getVal() {
        return val;
    }



}
