/**
 * Project Name:qiho-center-api
 * File Name:RemoteItemService.java
 * Package Name:com.qiho.center.api.remoteservice
 * Date:2017年5月18日上午9:37:17
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.params.item.ItemPageParams;

import java.util.List;

/**
 * ClassName:RemoteItemService <br/>
 * 类目管理dubbo服务 <br/>
 * Date:     2017年5月18日 上午9:37:17 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@AdvancedFeignClient
public interface RemoteItemService {

    /**
     * 分页查询商品列表
     *
     * @param queryItem
     * @param pageNum
     * @param pageSize
     * @return PagenationDto<ItemDto>
     * @author zhanglihui
     */
    DubboResult<PagenationDto<ItemDto>> queryItemPage(ItemDto queryItem, Long ownerId, Integer pageNum, Integer pageSize);

    /**
     * 新建一个商品
     *
     * @param itemDetail
     * @return 商品ID
     * @author zhanglihui
     */
    DubboResult<Long> createItem(ItemDetailDto itemDetail);

    /**
     * 更新商品信息
     *
     * @param itemDetail
     * @return DubboResult
     * @author zhanglihui
     */
    DubboResult<Void> updateItem(ItemDetailDto itemDetail);

    /**
     * 根据商品ID查询商品详情
     *
     * @param itemId
     * @return ItemDetailDto
     * @author zhanglihui
     */
    DubboResult<ItemDetailDto> queryItemDetail(Long itemId);

    /**
     * 批量下架商品
     *
     * @param itemIds 商品ID列表
     * @param status  更新后的商品状态，{@link ItemStatusEnum}
     * @return DubboResult
     * @author zhanglihui
     */
    DubboResult<Void> updateStatus(List<Long> itemIds, String status, Long gmtModifier);

    /**
     * 批量删除商品
     *
     * @param itemIds 商品ID列表
     * @return DubboResult
     * @author zhanglihui
     */
    DubboResult<Void> deleteBatch(List<Long> itemIds, Long gmtModifier);

    /**
     * 查询商品基本信息
     *
     * @param itemId 商品ID
     * @return ItemSimpleDto
     * @author zhanglihui
     */
    DubboResult<ItemSimpleDto> queryItemSimpleDto(Long itemId);

    /**
     * 不分页查询所有商品基本信息
     *
     * @return
     */
    DubboResult<List<ItemSimpleDto>> queryItemSimpleList(ItemDto queryItem);

    /**
     * 根据itemId,更新extParam扩展信息
     */
    DubboResult<Boolean> updateExtParamByItemId(ItemAdvertDto dto);

    /**
     * 根据itemId查询商品扩展信息
     */
    DubboResult<ItemAdvertDto> queryExtParamByItemId(Long itemId);

    /**
     * 根据商品id查询商品总库存
     */
    Long queryItemStockByItemId(Long itemId);

    /**
     * 更具商品id更新虚拟库存
     */
    Long updateVirtailStock(Long itemId);

    /**
     * 分页查询商品列表，用于外部商家
     *
     * @param params
     * @return
     */
    PagenationDto<ItemDto> queryMerchantItemPage(ItemPageParams params);


    /**
     * 根据商品id集合批量查询商品信息
     *
     * @param itemIdList
     * @return
     */
    List<ItemDto> queryItemsByItemIds(List<Long> itemIdList);

    /**
     * 当前商品Id 是否有促销价
     *
     * @param itemId
     * @return
     */
    boolean hasPromotionPrice(Long itemId);

    /**
     * 查询商品名字和id
     * @param merchantId 商铺id
     * @return
     */
    List<SimpleIdAndNameDto> queryByMerchantId(Long merchantId) throws BizException;

    /**
     * 查询商品的收银台图片
     *
     * @param itemId 商品id
     * @return 收银台图片
     */
    String queryCashierImg(Long itemId);
}

