package com.atlassian.net;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Enumeration;

/**
 * Various network-related utilities.
 */
public final class NetworkUtils
{
    private NetworkUtils() {}


    /**
     * <p>
     * Returns the fully-qualified hostname (FQHN) of the current machine on a best-effort basis.
     * Otherwise this method follows the same contract as {@link InetAddress#getHostName()}, with
     * the caveat that the value of {@link InetAddress#getCanonicalHostName()} will be returned
     * if the returned hostname of {@link InetAddress#getLocalHost()} is "localhost".
     * </p>
     * <p>
     * This method's main reason for existence is to shim the changed behaviour
     * of {@link InetAddress#getHostName()} when called on {@link InetAddress#getLocalHost()}
     * in Java8 compared to Java7, in which the former can return "localhost" instead of
     * the FQHN that was returned in Java7.
     * </p>
     */
    public static final String getLocalHostName() throws UnknownHostException
    {
        InetAddress host = InetAddress.getLocalHost();
        String hostname = host.getHostName();

        if (hostname != null && !hostname.equals("localhost"))
        {
            return hostname;
        }

        return host.getCanonicalHostName();
    }

    /**
     * <p>
     * Returns an <code>InetAddress</code> object containing the machine's LAN IP address.
     * </p>
     * <p>
     * Adapted from Brave <code>ThreadLocalServerClientAndLocalSpanState</code>, in turn adapted from https://issues.apache.org/jira/browse/JCS-40
     * </p>
     * @throws UnknownHostException
     */
    public static final InetAddress getLANAddress() throws UnknownHostException {
        try {
            InetAddress nonLoopbackNonSiteLocal = null;

            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();

            while (interfaces.hasMoreElements()) {
                NetworkInterface nic = interfaces.nextElement();
                Enumeration<InetAddress> addresses = nic.getInetAddresses();

                while (addresses.hasMoreElements()) {
                    InetAddress addr = addresses.nextElement();
                    if (!addr.isLoopbackAddress()) {
                        if (addr.isSiteLocalAddress()) {
                            return addr;
                        }
                        if (nonLoopbackNonSiteLocal == null) {
                            nonLoopbackNonSiteLocal = addr;
                        }
                    }
                }
            }

            if (nonLoopbackNonSiteLocal != null) {
                return nonLoopbackNonSiteLocal;
            }
            InetAddress fallbackAddress = InetAddress.getLocalHost();
            if (fallbackAddress == null) {
                throw new UnknownHostException();
            }
            return fallbackAddress;
        } catch (SocketException se) {
            throw new UnknownHostException();
        }
    }

    /**
     * <p>
     * Converts an <code>InetAddress</code> object to an integer representation of the encapsulated IP address
     * </p>
     * @param address encapsulates an IP address.
     * @return an integer representation of the IP address
     */
    public static final int addressToInt(InetAddress address) {
        return ByteBuffer.wrap(address.getAddress()).getInt();
    }

}
