package com.atlassian.plugin.webresource;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;

import java.util.Comparator;
import java.util.List;
import java.util.Set;

/**
 * A comparator that gives the order to batched resources.
 */
class BatchResourceComparator implements Comparator<PluginResource>
{
    static final Comparator<PluginResource> INSTANCE = new BatchResourceComparator();

    private final List<String> CONDITONAL_PARAMS = ImmutableList.of(
            PluginResourceLocator.CACHE_PARAM,
            PluginResourceLocator.MEDIA_PARAM,
            PluginResourceLocator.CONDITION_PARAM,
            PluginResourceLocator.IEONLY_PARAM);
    private final Set<String> CONDITONAL_PARAMS_SET = ImmutableSet.copyOf(CONDITONAL_PARAMS);

    @Override
    public int compare(PluginResource resource1, PluginResource resource2)
    {
        // the fewer conditions, the high priority
        Set<String> resource1Conditions = Sets.intersection(resource1.getParams().keySet(), CONDITONAL_PARAMS_SET);
        Set<String> resource2Conditions = Sets.intersection(resource2.getParams().keySet(), CONDITONAL_PARAMS_SET);

        // if only one condition, media comes first and ieonly comes last.
        if ((resource1Conditions.size() == 1) && (resource2Conditions.size() == 1))
        {
            String conditionOfResource1 = Iterables.getOnlyElement(resource1Conditions);
            String conditionOfResource2 = Iterables.getOnlyElement(resource2Conditions);

            int conditionIndexOfResource1 = CONDITONAL_PARAMS.indexOf(conditionOfResource1);
            int conditionIndexOfResource2 = CONDITONAL_PARAMS.indexOf(conditionOfResource2);

            if (conditionIndexOfResource1 != conditionIndexOfResource2)
            {
                return CONDITONAL_PARAMS.indexOf(conditionOfResource1) - CONDITONAL_PARAMS.indexOf(conditionOfResource2);
            }
        }

        if (resource1Conditions.size() == resource2Conditions.size())
        {
            return resource1.getUrl().compareTo(resource2.getUrl());
        }

        return resource1Conditions.size() - resource2Conditions.size();
    }
}