package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.util.EfficientStringUtils.endsWith;
import static com.atlassian.plugin.webresource.ContextBatchPluginResource.INCLUDE_SB_URL_PREFIX;
import static com.atlassian.plugin.webresource.ContextBatchPluginResource.REMOVE_SB_URL_PREFIX;
import static com.atlassian.plugin.webresource.SuperBatchPluginResource.DEFAULT_RESOURCE_NAME_PREFIX;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.servlet.DownloadableResource;

import com.atlassian.plugin.webresource.cache.CacheHandle;
import com.atlassian.plugin.webresource.cache.FileCacheKey;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;
import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Constructs a context batch resource for download
 * 
 * @since 2.9.0
 */
class ContextBatchDownloadableResourceBuilder extends AbstractBatchResourceBuilder
{
    private final ResourceDependencyResolver dependencyResolver;

    ContextBatchDownloadableResourceBuilder(final ResourceDependencyResolver dependencyResolver, final PluginAccessor pluginAccessor, final WebResourceUrlProvider webResourceUrlProvider,
                                            final DownloadableResourceFinder resourceFinder, final FileCache<FileCacheKey> cache,
                                            ResourceBatchingConfiguration resourceBatchingConfiguration)
    {
        super(pluginAccessor, webResourceUrlProvider, resourceFinder, cache, resourceBatchingConfiguration);
        this.dependencyResolver = dependencyResolver;
    }

    public boolean matches(final String path)
    {
        final String type = ResourceUtils.getType(path);
        boolean hasPrefix = path.contains(REMOVE_SB_URL_PREFIX + type) || path.contains(INCLUDE_SB_URL_PREFIX + type);
        return hasPrefix && endsWith(path, DEFAULT_RESOURCE_NAME_PREFIX, ".", type);
    }

    @Override
    public ContextBatchDownloadableResource parse(final String path, final Map<String, String> params)
    {
        final String type = ResourceUtils.getType(path);
        final boolean excludeSuperBatchedResources = path.contains(REMOVE_SB_URL_PREFIX + type);
        final String key = getKey(path);
        final LinkedHashSet<String> includedContexts = new LinkedHashSet<String>();
        final Set<String> excludedContexts = new HashSet<String>();
        final ConditionState conditionsRun = new ConditionState();

        ContextBatchOperations.parseContexts(key, includedContexts, excludedContexts);

        // Note that WebResourceModuleDescriptor equals method is based on module complete key,
        // so we eliminate dupes on module complete key through using a HashSet.
        LinkedHashSet<WebResourceModuleDescriptor> moduleDescriptors = new LinkedHashSet<WebResourceModuleDescriptor>();

        for (final String context : includedContexts)
        {
            Iterables.addAll(moduleDescriptors, dependencyResolver.getDependenciesInContext(context, excludeSuperBatchedResources, conditionsRun));
        }

        for (final String context : excludedContexts)
        {
            List<WebResourceModuleDescriptor> dependencies = new ArrayList<WebResourceModuleDescriptor>();
            Iterables.addAll(dependencies, dependencyResolver.getDependenciesInContext(context, excludeSuperBatchedResources, conditionsRun));

            moduleDescriptors.removeAll(dependencies);
        }

        Iterable<DownloadableResource> resources = Iterables.concat(Iterables.transform(moduleDescriptors, new Function<WebResourceModuleDescriptor, Iterable<DownloadableResource>>()
        {
            @Override
            public Iterable<DownloadableResource> apply(@Nullable WebResourceModuleDescriptor moduleDescriptor)
            {
                return resolve(moduleDescriptor, type, params);
            }
        }));

        final CacheHandle cacher = CacheHandle.Builder.forRequest(cache, "context", path, params);
        return new ContextBatchDownloadableResource(type, params, resources, cacher, resourceBatchingConfiguration);
    }

    private String getKey(final String path)
    {
        final int secondSlashIndex = path.lastIndexOf(PATH_SEPARATOR);
        final int firstSlashIndex = path.lastIndexOf(PATH_SEPARATOR, secondSlashIndex - 1);
        return path.substring(firstSlashIndex + 1, secondSlashIndex);
    }
}
