package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.webresource.ContextBatchPluginResource.INCLUDE_SB_URL_PREFIX;
import static com.atlassian.plugin.webresource.ContextBatchPluginResource.REMOVE_SB_URL_PREFIX;

import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Provides a fallback to serve resources relative to a context batch resource
 * In practice, the resources url should be transformed via the
 * {{com.atlassian.plugin.webresource.RelativeURLTransformResource}}.
 * This builder is in place in case this does not happen
 * @since 2.9.0
 */
public class ContextBatchSubResourceBuilder implements DownloadableResourceBuilder
{
    public static final Pattern CONTEXT_BATCH_PATTERN = Pattern.compile("(?:" + REMOVE_SB_URL_PREFIX + "|" + INCLUDE_SB_URL_PREFIX + ")\\w*/([^/]*)/(?!batch\\.js)(.*)$");

    private final ResourceDependencyResolver dependencyResolver;
    private final DownloadableResourceFinder resourceFinder;

    public ContextBatchSubResourceBuilder(final ResourceDependencyResolver dependencyResolver, final DownloadableResourceFinder resourceFinder)
    {
        this.dependencyResolver = dependencyResolver;
        this.resourceFinder = resourceFinder;
    }

    public boolean matches(final String path)
    {
        return CONTEXT_BATCH_PATTERN.matcher(path).find();
    }

    public DownloadableResource parse(final String path, final Map<String, String> params) throws UrlParseException
    {
        final Matcher m = CONTEXT_BATCH_PATTERN.matcher(path);
        if (!m.find())
        {
            throw new UrlParseException("Context batch url could not be parsed.");
        }

        final String key = m.group(1);
        final String resourceName = m.group(2);

        final LinkedHashSet<String> includedContexts = new LinkedHashSet<String>();
        final Set<String> excludedContexts = new HashSet<String>();
        final ConditionState conditionsRun = new ConditionState();

        ContextBatchOperations.parseContexts(key, includedContexts, excludedContexts);
        
        for (final String context : includedContexts)
        {
            for (final WebResourceModuleDescriptor moduleDescriptor : dependencyResolver.getDependenciesInContext(context, conditionsRun))
            {
                final DownloadableResource resource = resourceFinder.find(moduleDescriptor.getCompleteKey(), resourceName, params);
                if (resource != null)
                {
                    return resource;
                }
            }
        }
        return null; // 404 not found
    }
}