package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.CDNSupport;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.plugin.webresource.url.UrlParameters;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Represents a plugin resource.
 * This object represents a page-time resource. It is used build a URL to the underlying resource,
 * but does not know how to get the contents of that resource itself.
 *
 * @since 2.2
 */
public interface PluginResource
{
    /**
     * @return true if caching for this resource is supported. Use this check to append a static
     * caching url prefix to this resource's url.
     */
    boolean isCacheSupported();

    /**
     * @return the url for this plugin resource, never null.
     */
    String getUrl();

    /**
     * @return the url for this plugin resource, with it static prefix.
     */
    String getStaticUrl(WebResourceIntegration webResourceIntegration, WebResourceUrlProvider webResourceUrlProvider, UrlMode urlMode);

    /**
     * @return the resource name for the plugin resource.
     */
    String getResourceName();

    /**
     * @return the plugin module's complete key for which this resource belongs to.
     */
    String getModuleCompleteKey();

    /**
     * @return all contributed hashes to this url
     */
    List<NamedHash> getHashes();

    /**
     * @return url parameters
     */
    UrlParameters getUrlParameters();

    CDNSupport supportsCdn();

    /**
     * @return a map of parameter key and value pairs for this resource, can be empty but never null. Resources that
     * have any values in this map will be included in a separate batch.
     */
    Map<String, String> getParams();

    /**
     * @return a map of parameter key and value pairs for this resource, can be empty but never null. Including
     * a resource in this map will not cause the resource to be excluded from a batch.
     */
    Map<String, String> getBatchableParams();

    /**
     * @return the version prefix string for a cached resource
     */
    String getVersion(WebResourceIntegration integration);

    /**
     * @return the type (e.g. file extension) of the resource.
     * @since 2.12.3
     */
    String getType();
    
    /**
     * This method provides the mechanism for discovering the contents of a PluginResource.
     * <p/>
     * The primary use of this method is in providing a representation of the contents 
     * (for dependency tracking) when formatting a PluginResource.
     * 
     * @return a Set of descriptors providing information about the contents of this PluginResource.
     */
    Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors();
}