package com.atlassian.plugin.webresource;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.webresource.api.data.PluginDataResource;

import java.util.List;
import java.util.Map;

/**
 * Assists in locating plugin resources in different ways.  
 * @since 2.2
 */
public interface PluginResourceLocator
{
    static final String IEONLY_PARAM = "ieonly";
    static final String MEDIA_PARAM = "media";
    static final String CONTENT_TYPE_PARAM = "content-type";
    static final String CACHE_PARAM = "cache";
    static final String CONDITION_PARAM = "conditionalComment";

    String[] BATCH_PARAMS = new String[] { IEONLY_PARAM, MEDIA_PARAM, CONTENT_TYPE_PARAM, CACHE_PARAM, CONDITION_PARAM };

    /**
     * Returns true if this locator can parse the given url.
     */
    boolean matches(String url);

    /**
     * Returns a {@link DownloadableResource} represented by the given url and query params.
     * {@link #matches(String)} should be called before invoking this method. If the url is
     * not understood by the locator or the resource cannot be found, null will be returned.
     */
    DownloadableResource getDownloadableResource(String url, Map<String, String> queryParams);

    /**
     * Returns a list of {@link PluginDataResource}s for a given plugin module's complete key. If
     * the plugin the module belongs to is not enabled or does not exist, an empty list is returned.
     */
    Iterable<PluginDataResource> getPluginDataResources(String moduleCompleteKey);

    /**
     * Returns a list of {@link PluginDataResource}s for a given plugin module descriptor. If
     * the plugin the module belongs to is not enabled or does not exist, an empty list is returned.
     * @since 3.2
     */
    Iterable<PluginDataResource> getPluginDataResources(WebResourceModuleDescriptor webResourceModuleDescriptor);

    /**
     * Returns a list of {@link PluginResource}s for a given plugin module's complete key. If
     * the plugin the module belongs to is not enabled or does not exist, an empty list is returned.
     */
    List<PluginResource> getPluginResources(String moduleCompleteKey, ConditionState conditionsRun);

    /**
     * Returns a list of {@link PluginResource}s for a given plugin module's complete key. If
     * the plugin the module belongs to is not enabled or does not exist, an empty list is returned.
     * @since 3.2
     */
    List<PluginResource> getPluginResources(WebResourceModuleDescriptor webResourceModuleDescriptor, ConditionState conditionsRun);

    /**
     * Gets the module descriptor for the given complete key. If the plugin the module belongs to is not enabled or does
     * not exist, returns None.
     * @param moduleCompleteKey the plugin and module key
     * @return Some WebResourceModuleDescriptor, if it exists and is enabled. Otherwise, None.
     * @since 3.2
     */
    Option<WebResourceModuleDescriptor> getDescriptor(String moduleCompleteKey);

    /**
     * Returns TransformerCache, instantiating it here instead of making it component in order to
     * avoid its explicit declaration in JIRA in Pico Container.
     */
    @com.atlassian.annotations.Internal
    TransformerCache getTransformerCache();
}